# 安全指南

## 🔒 安全特性

### 1. IP过滤

```python
from server.core.security.ip_filter import IPFilter

# 创建IP过滤器
ip_filter = IPFilter()

# 添加白名单
ip_filter.add_to_whitelist("192.168.1.0/24")

# 添加黑名单
ip_filter.add_to_blacklist("10.0.0.1")

# 检查IP
if ip_filter.is_allowed("192.168.1.100"):
    # 允许访问
    pass
```

### 2. 速率限制

```python
from server.core.security.api_rate_limiter import APIRateLimiter

# 创建限流器
rate_limiter = APIRateLimiter()

# 添加限流规则
rate_limiter.add_limit("api/clients", max_requests=100, window_seconds=60)

# 检查限流
if rate_limiter.check_limit("api/clients"):
    # 允许请求
    pass
```

### 3. 熔断器

```python
from server.core.security.circuit_breaker import CircuitBreaker, CircuitBreakerConfig

# 创建熔断器
config = CircuitBreakerConfig(
    failure_threshold=5,
    success_threshold=2,
    timeout=60.0
)
breaker = CircuitBreaker("api_service", config)

# 使用熔断器保护函数调用
try:
    result = breaker.call(risky_function)
except Exception as e:
    # 处理熔断或异常
    pass
```

### 4. 异常检测

```python
from server.core.security.anomaly_detector import AnomalyDetector

# 创建异常检测器
detector = AnomalyDetector()

# 检测异常
if detector.detect_anomaly(request_data):
    # 触发告警
    pass
```

## 🛡️ DDoS防护

### 配置DDoS防护

```yaml
# config.yaml
security:
  ddos_protection:
    enabled: true
    max_requests_per_second: 100
    block_duration_seconds: 3600
    whitelist:
      - "192.168.1.0/24"
```

### 自动防护机制

- 自动检测异常流量模式
- 自动封禁攻击IP
- 自动触发限流
- 自动发送告警

## 🔐 API认证

### JWT Token认证

```python
from server.core.api_auth.api_auth import APIAuth, Permission, Role

# 创建API认证管理器
api_auth = APIAuth(secret_key="your-secret-key")

# 创建Token
token = api_auth.create_api_token("username", Role.ADMIN)

# 验证Token
payload = api_auth.verify_token(token)
```

### 权限控制

```python
from fastapi import Depends
from server.core.api_auth.api_auth import APIAuth, Permission

# 使用权限装饰器
@router.get("/api/clients")
async def get_clients(
    user: dict = Depends(api_auth.require_permission(Permission.CLIENT_VIEW))
):
    # 只有有CLIENT_VIEW权限的用户可以访问
    pass
```

## 📋 安全最佳实践

### 1. 密钥管理

- ✅ 使用环境变量存储密钥
- ✅ 定期轮换密钥
- ✅ 不要在代码中硬编码密钥
- ✅ 使用密钥管理服务（如Vault）

### 2. HTTPS配置

```nginx
# Nginx配置
server {
    listen 443 ssl;
    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
}
```

### 3. 输入验证

```python
from pydantic import BaseModel, validator

class ClientRequest(BaseModel):
    name: str
    
    @validator('name')
    def validate_name(cls, v):
        if len(v) < 3 or len(v) > 100:
            raise ValueError('名称长度必须在3-100字符之间')
        return v
```

### 4. SQL注入防护

- ✅ 使用ORM（SQLAlchemy）
- ✅ 使用参数化查询
- ✅ 验证和清理用户输入

### 5. XSS防护

- ✅ 转义用户输入
- ✅ 使用CSP（Content Security Policy）
- ✅ 验证和清理HTML内容

## 🚨 安全告警

### 告警规则

- CPU使用率 > 80%
- 内存使用率 > 85%
- 错误率突然增加
- DDoS攻击检测
- 异常登录尝试
- 熔断器打开

### 告警通知

- 邮件通知
- Webhook通知
- 短信通知（可选）
- 钉钉/企业微信通知（可选）

## 📊 安全监控

### 监控指标

- `nps_security_blocked_ips_total` - 被阻止的IP数
- `nps_security_anomalies_total` - 安全异常数
- `nps_rate_limit_hits_total` - 限流命中数
- `nps_circuit_breaker_state` - 熔断器状态

### 日志审计

```python
# 记录安全事件
logger.security("IP blocked", extra={
    'ip': client_ip,
    'reason': 'blacklist',
    'timestamp': datetime.now()
})
```

## 🔍 安全检查清单

- [x] ✅ **启用HTTPS** - 自动配置和证书管理
- [x] ✅ **配置防火墙规则** - 防火墙管理器
- [x] ✅ **启用IP白名单/黑名单** - IP过滤器
- [x] ✅ **配置速率限制** - API限流器
- [x] ✅ **启用异常检测** - 异常检测器
- [x] ✅ **配置告警规则** - 告警规则引擎
- [x] ✅ **定期安全审计** - 安全审计器
- [x] ✅ **密钥轮换** - 密钥轮换管理器
- [x] ✅ **日志审计** - 审计日志记录器
- [x] ✅ **备份加密** - 加密备份功能

## 🛠️ 安全功能使用

### HTTPS自动配置

```python
from server.core.security.https_manager import HTTPSManager

https_manager = HTTPSManager()
https_manager.enable_https("example.com", email="admin@example.com")

# 获取SSL上下文
ssl_context = https_manager.get_ssl_context()
```

### 防火墙配置

```python
from server.core.security.firewall_manager import FirewallManager, FirewallRule, FirewallAction

firewall = FirewallManager()
firewall.enable()

# 添加规则
firewall.add_rule(FirewallRule(
    name="allow_api",
    action=FirewallAction.ALLOW,
    protocol="tcp",
    port=8080,
    source_ip="192.168.1.0/24"
))

# 检查访问权限
if firewall.check_access("tcp", 8080, "192.168.1.100"):
    # 允许访问
    pass
```

### 安全审计

```python
from server.core.security.security_audit import SecurityAuditor

auditor = SecurityAuditor()
results = auditor.run_full_audit()

# 获取最新审计结果
latest_audit = auditor.get_latest_audit()
```

### 密钥轮换

```python
from server.core.security.key_rotation import KeyRotationManager, SecretKeyRotator

# 创建密钥管理器
key_manager = KeyRotationManager()

# 生成新密钥
new_key = key_manager.generate_key("jwt_secret")

# 轮换密钥
rotated_key = key_manager.rotate_key("jwt_secret")

# 检查即将过期的密钥
expiring = key_manager.check_expiring_keys(days_before=7)

# 自动轮换过期密钥
rotated = key_manager.auto_rotate_expired_keys()
```

### 日志审计

```python
from server.core.logging.audit_logger import AuditLogger, AuditEventType

audit_logger = AuditLogger()

# 记录认证事件
audit_logger.log_authentication(
    user_id="user123",
    client_ip="192.168.1.100",
    status="success"
)

# 记录授权事件
audit_logger.log_authorization(
    user_id="user123",
    action="create_client",
    resource="/api/clients",
    status="success"
)

# 查询审计日志
logs = audit_logger.query_logs(
    start_time=datetime.now() - timedelta(days=7),
    event_type=AuditEventType.AUTHENTICATION,
    limit=100
)

# 获取统计信息
stats = audit_logger.get_statistics(days=7)
```

### 备份加密

```python
from server.core.backup.backup_manager import BackupManager

# 创建启用加密的备份管理器
backup_manager = BackupManager(encryption_key="your-encryption-password")

# 创建加密备份
backup_path = backup_manager.create_backup(encrypt=True)

# 恢复加密备份
backup_manager.restore_backup("backup_name")
```

