# 性能优化指南

## 📊 性能基准测试

### 运行基准测试

```bash
# 运行性能基准测试
python -m pytest tests/performance/benchmark.py -v

# 或直接运行
python tests/performance/benchmark.py
```

### 基准测试指标

- **API响应时间**: P50, P95, P99
- **并发处理能力**: QPS (Queries Per Second)
- **WebSocket连接性能**: 连接建立时间
- **隧道吞吐量**: MB/s, packets/s

## 🔥 压力测试

### 运行压力测试

```bash
# 运行压力测试
python tests/performance/stress_test.py

# 自定义参数
python tests/performance/stress_test.py --duration 300 --rate 200
```

### 压力测试场景

1. **API压力测试**: 高并发API请求
2. **WebSocket压力测试**: 大量并发连接
3. **混合工作负载**: API + WebSocket混合场景

## ⚡ 性能优化建议

### 1. 数据库优化

```python
# 使用连接池
from sqlalchemy.pool import QueuePool

engine = create_engine(
    DATABASE_URL,
    poolclass=QueuePool,
    pool_size=20,
    max_overflow=40,
    pool_pre_ping=True
)
```

### 2. 缓存优化

```python
# 使用Redis缓存热点数据
from server.core.cache.cache_manager import CacheManager

cache = CacheManager()
cache.set("key", "value", ttl=3600)
```

### 3. 异步处理

```python
# 使用异步任务队列处理耗时操作
import asyncio

async def process_background_task():
    # 后台任务
    pass

asyncio.create_task(process_background_task())
```

### 4. 连接复用

```python
# 复用HTTP客户端连接
async with aiohttp.ClientSession() as session:
    # 多个请求复用同一个session
    pass
```

## 📈 监控指标

### Prometheus指标

- `nps_http_request_duration_seconds` - HTTP请求耗时
- `nps_connections_active` - 活跃连接数
- `nps_traffic_bytes_total` - 流量统计
- `nps_system_cpu_percent` - CPU使用率
- `nps_system_memory_bytes` - 内存使用

### Grafana仪表盘

导入Prometheus数据源，创建自定义仪表盘：

```yaml
# grafana/dashboard.yaml
- 连接数监控
- 响应时间监控
- 流量统计
- 系统资源监控
- 错误率监控
```

## 🎯 性能调优清单

- [ ] 数据库连接池配置
- [ ] Redis缓存策略
- [ ] 异步任务队列
- [ ] 连接复用
- [ ] 请求批处理
- [ ] 数据压缩
- [ ] CDN加速（静态资源）
- [ ] 负载均衡配置

## 📚 参考资源

- [FastAPI性能优化](https://fastapi.tiangolo.com/advanced/performance/)
- [Python异步编程最佳实践](https://docs.python.org/3/library/asyncio.html)
- [PostgreSQL性能调优](https://www.postgresql.org/docs/current/performance-tips.html)

