# 使用示例

## 📚 API使用示例

### Python客户端

```python
from examples.api_client_example import NPSAPIClient
import asyncio

async def main():
    client = NPSAPIClient(
        base_url="http://localhost:8080",
        api_token="your-token"
    )
    
    # 创建客户端
    new_client = await client.create_client("My Client")
    
    # 创建隧道
    tunnel = await client.create_tunnel(new_client["id"], {
        "name": "SSH Tunnel",
        "tunnel_type": "tcp",
        "local_host": "127.0.0.1",
        "local_port": 22,
        "remote_port": 2222
    })

asyncio.run(main())
```

### cURL示例

```bash
# 创建客户端
curl -X POST http://localhost:8080/api/clients \
  -H "Content-Type: application/json" \
  -d '{"name": "My Client"}'

# 获取客户端列表
curl http://localhost:8080/api/clients

# 创建隧道
curl -X POST http://localhost:8080/api/clients/{client_id}/tunnels \
  -H "Content-Type: application/json" \
  -d '{
    "name": "SSH Tunnel",
    "tunnel_type": "tcp",
    "local_host": "127.0.0.1",
    "local_port": 22,
    "remote_port": 2222
  }'
```

## 🔌 WebSocket客户端示例

```python
from examples.websocket_client_example import NPSWebSocketClient
import asyncio

async def main():
    client = NPSWebSocketClient(
        ws_url="ws://localhost:8080",
        client_id="your-client-id",
        token="your-token"
    )
    
    if await client.connect():
        await client.listen()

asyncio.run(main())
```

## 🛡️ 安全功能示例

### IP白名单

```python
from server.core.security.ip_filter import IPFilter

ip_filter = IPFilter()
ip_filter.add_to_whitelist("192.168.1.0/24")

if ip_filter.is_allowed("192.168.1.100"):
    # 允许访问
    pass
```

### 速率限制

```python
from server.core.security.api_rate_limiter import APIRateLimiter

rate_limiter = APIRateLimiter()
rate_limiter.add_limit("api/clients", max_requests=100, window_seconds=60)

if rate_limiter.check_limit("api/clients"):
    # 允许请求
    pass
```

### 熔断器

```python
from server.core.security.circuit_breaker import CircuitBreaker, CircuitBreakerConfig

config = CircuitBreakerConfig(
    failure_threshold=5,
    success_threshold=2,
    timeout=60.0
)
breaker = CircuitBreaker("api_service", config)

try:
    result = breaker.call(risky_function)
except Exception as e:
    # 处理熔断或异常
    pass
```

## 📊 监控示例

### Prometheus指标

```python
from server.core.monitoring.prometheus_metrics import PrometheusMetrics

metrics = PrometheusMetrics()

# 记录连接
metrics.record_connection("connected")

# 记录流量
metrics.record_traffic(1024, "inbound", "client-1", "tunnel-1")

# 记录HTTP请求
metrics.record_http_request("GET", "/api/clients", 200, 0.1)
```

### 告警规则

```python
from server.core.alerting.alert_rules import AlertRuleEngine
from server.core.alerting.alert_manager import AlertManager

alert_manager = AlertManager()
rule_engine = AlertRuleEngine(alert_manager)

# 评估规则
rule_engine.evaluate_rule("high_cpu_usage", cpu_percent=85.0)
```

## 🚀 性能测试示例

### 基准测试

```python
from tests.performance.benchmark import PerformanceBenchmark
import asyncio

async def main():
    benchmark = PerformanceBenchmark()
    await benchmark.benchmark_api_request("/api/health", iterations=100)
    await benchmark.benchmark_concurrent_requests("/api/health", concurrency=100)

asyncio.run(main())
```

### 压力测试

```python
from tests.performance.stress_test import StressTest
import asyncio

async def main():
    stress_test = StressTest()
    await stress_test.stress_test_api("/api/health", duration=60, rate=100)

asyncio.run(main())
```

## 📝 更多示例

查看 `examples/` 目录获取更多示例代码。

