"""
连接管理器测试
"""
import pytest
import asyncio
from unittest.mock import Mock, AsyncMock, patch

from server.core.gateway.connection_manager import ConnectionManager, ClientConnection


@pytest.fixture
def connection_manager():
    """创建连接管理器实例"""
    return ConnectionManager()


@pytest.fixture
def mock_websocket():
    """创建模拟WebSocket"""
    ws = AsyncMock()
    ws.send = AsyncMock()
    ws.recv = AsyncMock()
    ws.close = AsyncMock()
    return ws


@pytest.mark.asyncio
async def test_add_connection(connection_manager, mock_websocket):
    """测试添加连接"""
    connection = ClientConnection(
        websocket=mock_websocket,
        remote_addr="127.0.0.1",
        protocol="websocket"
    )
    connection.client_id = "test_client"
    
    result = await connection_manager.add_connection(connection)
    assert result is True
    assert connection_manager.is_connected("test_client")
    assert connection_manager.get_connection_count() == 1


@pytest.mark.asyncio
async def test_remove_connection(connection_manager, mock_websocket):
    """测试移除连接"""
    connection = ClientConnection(
        websocket=mock_websocket,
        remote_addr="127.0.0.1",
        protocol="websocket"
    )
    connection.client_id = "test_client"
    
    await connection_manager.add_connection(connection)
    await connection_manager.remove_connection("test_client")
    
    assert not connection_manager.is_connected("test_client")
    assert connection_manager.get_connection_count() == 0


@pytest.mark.asyncio
async def test_max_connections(connection_manager, mock_websocket):
    """测试最大连接数限制"""
    # 设置较小的最大连接数用于测试
    connection_manager._max_connections = 2
    
    # 添加两个连接
    for i in range(2):
        connection = ClientConnection(
            websocket=mock_websocket,
            remote_addr=f"127.0.0.{i}",
            protocol="websocket"
        )
        connection.client_id = f"client_{i}"
        await connection_manager.add_connection(connection)
    
    # 尝试添加第三个连接应该失败
    connection3 = ClientConnection(
        websocket=mock_websocket,
        remote_addr="127.0.0.3",
        protocol="websocket"
    )
    connection3.client_id = "client_3"
    result = await connection_manager.add_connection(connection3)
    
    assert result is False
    assert connection_manager.get_connection_count() == 2

