"""
性能基准测试
"""
import asyncio
import time
import statistics
from typing import List, Dict
from loguru import logger
import aiohttp
import websockets


class PerformanceBenchmark:
    """性能基准测试"""
    
    def __init__(self, base_url: str = "http://localhost:8080"):
        self.base_url = base_url
        self.ws_url = base_url.replace("http", "ws")
        self.results: Dict[str, List[float]] = {}
    
    async def benchmark_api_request(self, endpoint: str, method: str = "GET", 
                                   data: Dict = None, iterations: int = 100):
        """API请求性能测试"""
        logger.info(f"开始API性能测试: {method} {endpoint} ({iterations}次)")
        
        times = []
        async with aiohttp.ClientSession() as session:
            for i in range(iterations):
                start_time = time.time()
                
                try:
                    if method == "GET":
                        async with session.get(f"{self.base_url}{endpoint}") as resp:
                            await resp.read()
                    elif method == "POST":
                        async with session.post(f"{self.base_url}{endpoint}", json=data) as resp:
                            await resp.read()
                    elif method == "PUT":
                        async with session.put(f"{self.base_url}{endpoint}", json=data) as resp:
                            await resp.read()
                    elif method == "DELETE":
                        async with session.delete(f"{self.base_url}{endpoint}") as resp:
                            await resp.read()
                    
                    elapsed = time.time() - start_time
                    times.append(elapsed)
                except Exception as e:
                    logger.error(f"请求失败: {e}")
        
        if times:
            self.results[f"{method}_{endpoint}"] = times
            self._print_stats(f"{method} {endpoint}", times)
    
    async def benchmark_websocket_connection(self, iterations: int = 50):
        """WebSocket连接性能测试"""
        logger.info(f"开始WebSocket连接性能测试 ({iterations}次)")
        
        times = []
        for i in range(iterations):
            start_time = time.time()
            
            try:
                async with websockets.connect(f"{self.ws_url}/ws/client") as ws:
                    await ws.recv()
                    elapsed = time.time() - start_time
                    times.append(elapsed)
                    await ws.close()
            except Exception as e:
                logger.error(f"WebSocket连接失败: {e}")
        
        if times:
            self.results["websocket_connection"] = times
            self._print_stats("WebSocket连接", times)
    
    async def benchmark_concurrent_requests(self, endpoint: str, 
                                          concurrency: int = 100, 
                                          total_requests: int = 1000):
        """并发请求性能测试"""
        logger.info(f"开始并发性能测试: {endpoint} (并发数: {concurrency}, 总请求: {total_requests})")
        
        async def make_request(session, semaphore):
            async with semaphore:
                start_time = time.time()
                try:
                    async with session.get(f"{self.base_url}{endpoint}") as resp:
                        await resp.read()
                    return time.time() - start_time
                except Exception as e:
                    logger.error(f"请求失败: {e}")
                    return None
        
        semaphore = asyncio.Semaphore(concurrency)
        times = []
        
        async with aiohttp.ClientSession() as session:
            tasks = [make_request(session, semaphore) for _ in range(total_requests)]
            results = await asyncio.gather(*tasks)
            times = [t for t in results if t is not None]
        
        if times:
            self.results[f"concurrent_{endpoint}"] = times
            self._print_stats(f"并发请求 ({concurrency}并发)", times)
    
    async def benchmark_tunnel_throughput(self, tunnel_id: str, 
                                         data_size: int = 1024,
                                         duration: int = 60):
        """隧道吞吐量测试"""
        logger.info(f"开始隧道吞吐量测试: {tunnel_id} (数据大小: {data_size}字节, 持续时间: {duration}秒)")
        
        data = b"x" * data_size
        start_time = time.time()
        bytes_sent = 0
        packets_sent = 0
        
        try:
            async with websockets.connect(f"{self.ws_url}/ws/client") as ws:
                # 认证（如果需要）
                # await ws.send(...)
                
                while time.time() - start_time < duration:
                    await ws.send(data)
                    bytes_sent += len(data)
                    packets_sent += 1
                    await asyncio.sleep(0.01)  # 避免过快发送
                
                elapsed = time.time() - start_time
                throughput = bytes_sent / elapsed
                pps = packets_sent / elapsed
                
                logger.info(f"隧道吞吐量测试结果:")
                logger.info(f"  总数据量: {bytes_sent / 1024 / 1024:.2f} MB")
                logger.info(f"  吞吐量: {throughput / 1024 / 1024:.2f} MB/s")
                logger.info(f"  包速率: {pps:.2f} packets/s")
                
        except Exception as e:
            logger.error(f"隧道吞吐量测试失败: {e}")
    
    def _print_stats(self, name: str, times: List[float]):
        """打印统计信息"""
        if not times:
            return
        
        avg = statistics.mean(times)
        median = statistics.median(times)
        min_time = min(times)
        max_time = max(times)
        p95 = self._percentile(times, 95)
        p99 = self._percentile(times, 99)
        
        logger.info(f"{name} 性能统计:")
        logger.info(f"  平均: {avg*1000:.2f}ms")
        logger.info(f"  中位数: {median*1000:.2f}ms")
        logger.info(f"  最小: {min_time*1000:.2f}ms")
        logger.info(f"  最大: {max_time*1000:.2f}ms")
        logger.info(f"  P95: {p95*1000:.2f}ms")
        logger.info(f"  P99: {p99*1000:.2f}ms")
        logger.info(f"  QPS: {1/avg:.2f}")
    
    def _percentile(self, data: List[float], percentile: int) -> float:
        """计算百分位数"""
        sorted_data = sorted(data)
        index = int(len(sorted_data) * percentile / 100)
        return sorted_data[min(index, len(sorted_data) - 1)]
    
    def get_summary(self) -> Dict:
        """获取测试摘要"""
        summary = {}
        for name, times in self.results.items():
            if times:
                summary[name] = {
                    'avg': statistics.mean(times),
                    'median': statistics.median(times),
                    'min': min(times),
                    'max': max(times),
                    'p95': self._percentile(times, 95),
                    'p99': self._percentile(times, 99),
                    'qps': 1 / statistics.mean(times),
                }
        return summary


async def run_benchmark():
    """运行性能基准测试"""
    benchmark = PerformanceBenchmark()
    
    # API性能测试
    await benchmark.benchmark_api_request("/api/health", "GET", iterations=100)
    await benchmark.benchmark_api_request("/api/clients", "GET", iterations=50)
    
    # 并发测试
    await benchmark.benchmark_concurrent_requests("/api/health", concurrency=100, total_requests=1000)
    
    # WebSocket连接测试
    await benchmark.benchmark_websocket_connection(iterations=50)
    
    # 打印摘要
    summary = benchmark.get_summary()
    logger.info("性能测试摘要:")
    for name, stats in summary.items():
        logger.info(f"{name}: {stats}")


if __name__ == "__main__":
    asyncio.run(run_benchmark())

