"""
API限流集成测试
"""
import pytest
import asyncio
import aiohttp
from fastapi.testclient import TestClient


@pytest.fixture
def client():
    """创建测试客户端"""
    from server.main import app
    return TestClient(app)


def test_rate_limit_basic(client):
    """测试基本限流功能"""
    # 发送大量请求
    responses = []
    for i in range(150):  # 超过默认100的限制
        response = client.get("/api/health")
        responses.append(response.status_code)
    
    # 应该有一些请求被限流（429）
    assert 429 in responses, "应该有请求被限流"
    
    # 检查限流头信息
    response = client.get("/api/health")
    assert "X-RateLimit-Limit" in response.headers
    assert "X-RateLimit-Remaining" in response.headers


def test_rate_limit_per_endpoint(client):
    """测试不同端点的限流"""
    # 不同端点应该有独立的限流
    health_responses = [client.get("/api/health").status_code for _ in range(150)]
    clients_responses = [client.get("/api/clients").status_code for _ in range(150)]
    
    # 两个端点都应该有独立的限流
    assert 429 in health_responses or 429 in clients_responses


@pytest.mark.asyncio
async def test_rate_limit_concurrent():
    """测试并发请求的限流"""
    async def make_request(session, url):
        async with session.get(url) as resp:
            return resp.status
    
    async with aiohttp.ClientSession() as session:
        url = "http://localhost:8080/api/health"
        tasks = [make_request(session, url) for _ in range(200)]
        statuses = await asyncio.gather(*tasks)
        
        # 应该有一些请求被限流
        assert 429 in statuses


def test_rate_limit_reset(client):
    """测试限流重置"""
    # 发送请求直到被限流
    responses = []
    for i in range(150):
        response = client.get("/api/health")
        responses.append(response.status_code)
        if response.status_code == 429:
            break
    
    # 等待一段时间后，应该可以继续请求
    import time
    time.sleep(2)  # 等待限流窗口重置
    
    # 应该可以继续请求
    response = client.get("/api/health")
    assert response.status_code in [200, 429]  # 可能还在限流窗口内

