"""
熔断器集成测试
"""
import pytest
import time
from server.core.security.circuit_breaker import CircuitBreaker, CircuitBreakerConfig, CircuitState


def test_circuit_breaker_closed_state():
    """测试熔断器关闭状态"""
    breaker = CircuitBreaker("test", CircuitBreakerConfig(failure_threshold=3))
    
    assert breaker.state == CircuitState.CLOSED
    
    # 正常执行
    def success_func():
        return "success"
    
    result = breaker.call(success_func)
    assert result == "success"
    assert breaker.state == CircuitState.CLOSED


def test_circuit_breaker_open_state():
    """测试熔断器打开状态"""
    config = CircuitBreakerConfig(failure_threshold=3, timeout=1.0)
    breaker = CircuitBreaker("test", config)
    
    # 模拟失败
    def fail_func():
        raise Exception("test error")
    
    # 触发失败阈值
    for i in range(3):
        try:
            breaker.call(fail_func)
        except Exception:
            pass
    
    # 应该进入打开状态
    assert breaker.state == CircuitState.OPEN
    
    # 打开状态下应该拒绝执行
    with pytest.raises(Exception, match="拒绝执行"):
        breaker.call(lambda: "should not execute")


def test_circuit_breaker_half_open_state():
    """测试熔断器半开状态"""
    config = CircuitBreakerConfig(
        failure_threshold=3,
        success_threshold=2,
        timeout=0.1
    )
    breaker = CircuitBreaker("test", config)
    
    # 触发熔断
    def fail_func():
        raise Exception("test error")
    
    for i in range(3):
        try:
            breaker.call(fail_func)
        except Exception:
            pass
    
    assert breaker.state == CircuitState.OPEN
    
    # 等待超时，进入半开状态
    time.sleep(0.2)
    
    # 尝试执行（应该允许）
    def success_func():
        return "success"
    
    # 在半开状态下成功执行两次
    result1 = breaker.call(success_func)
    assert result1 == "success"
    
    result2 = breaker.call(success_func)
    assert result2 == "success"
    
    # 应该恢复到关闭状态
    assert breaker.state == CircuitState.CLOSED


def test_circuit_breaker_reset():
    """测试熔断器重置"""
    breaker = CircuitBreaker("test")
    
    # 触发失败
    def fail_func():
        raise Exception("test error")
    
    for i in range(5):
        try:
            breaker.call(fail_func)
        except Exception:
            pass
    
    # 重置
    breaker.reset()
    
    assert breaker.state == CircuitState.CLOSED
    assert breaker.failure_count == 0
    assert breaker.success_count == 0


@pytest.mark.asyncio
async def test_circuit_breaker_async():
    """测试异步熔断器"""
    breaker = CircuitBreaker("test")
    
    async def async_success():
        await asyncio.sleep(0.01)
        return "success"
    
    result = await breaker.call_async(async_success)
    assert result == "success"
    
    async def async_fail():
        await asyncio.sleep(0.01)
        raise Exception("async error")
    
    # 触发失败
    for i in range(5):
        try:
            await breaker.call_async(async_fail)
        except Exception:
            pass
    
    assert breaker.state == CircuitState.OPEN

