// WebSocket实时推送客户端
class WebSocketClient {
    constructor(url) {
        this.url = url;
        this.ws = null;
        this.reconnectInterval = 5000;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = -1; // -1表示无限重连
        this.listeners = {};
    }
    
    connect() {
        try {
            this.ws = new WebSocket(this.url);
            
            this.ws.onopen = () => {
                console.log('WebSocket连接已建立');
                this.reconnectAttempts = 0;
                this.emit('connect');
            };
            
            this.ws.onmessage = (event) => {
                try {
                    const data = JSON.parse(event.data);
                    this.emit('message', data);
                    // 根据消息类型分发
                    if (data.type) {
                        this.emit(data.type, data);
                    }
                } catch (e) {
                    console.error('解析WebSocket消息失败:', e);
                }
            };
            
            this.ws.onerror = (error) => {
                console.error('WebSocket错误:', error);
                this.emit('error', error);
            };
            
            this.ws.onclose = () => {
                console.log('WebSocket连接已关闭');
                this.emit('disconnect');
                this.reconnect();
            };
            
        } catch (e) {
            console.error('WebSocket连接失败:', e);
            this.reconnect();
        }
    }
    
    reconnect() {
        if (this.maxReconnectAttempts > 0 && this.reconnectAttempts >= this.maxReconnectAttempts) {
            console.error('达到最大重连次数，停止重连');
            return;
        }
        
        this.reconnectAttempts++;
        console.log(`等待 ${this.reconnectInterval}ms 后重连... (尝试 ${this.reconnectAttempts})`);
        
        setTimeout(() => {
            this.connect();
        }, this.reconnectInterval);
    }
    
    on(event, callback) {
        if (!this.listeners[event]) {
            this.listeners[event] = [];
        }
        this.listeners[event].push(callback);
    }
    
    off(event, callback) {
        if (this.listeners[event]) {
            this.listeners[event] = this.listeners[event].filter(cb => cb !== callback);
        }
    }
    
    emit(event, data) {
        if (this.listeners[event]) {
            this.listeners[event].forEach(callback => {
                try {
                    callback(data);
                } catch (e) {
                    console.error(`事件处理器错误 [${event}]:`, e);
                }
            });
        }
    }
    
    send(data) {
        if (this.ws && this.ws.readyState === WebSocket.OPEN) {
            this.ws.send(JSON.stringify(data));
        } else {
            console.warn('WebSocket未连接，无法发送消息');
        }
    }
    
    disconnect() {
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }
    }
}

// 全局WebSocket客户端
let wsClient = null;

// 初始化WebSocket连接
function initWebSocket() {
    const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
    const wsUrl = `${protocol}//${window.location.host}/api/ws/admin`;
    
    wsClient = new WebSocketClient(wsUrl);
    
    // 监听连接事件
    wsClient.on('connect', () => {
        console.log('管理面板WebSocket已连接');
        updateConnectionStatus(true);
    });
    
    wsClient.on('disconnect', () => {
        console.log('管理面板WebSocket已断开');
        updateConnectionStatus(false);
    });
    
    // 监听实时更新
    wsClient.on('stats_update', (data) => {
        console.log('收到统计更新:', data);
        // 刷新统计信息
        if (currentPage === 'dashboard') {
            loadPage();
        }
    });
    
    wsClient.on('client_update', (data) => {
        console.log('收到客户端更新:', data);
        // 刷新客户端列表
        if (currentPage === 'clients') {
            loadPage();
        }
    });
    
    wsClient.on('tunnel_update', (data) => {
        console.log('收到隧道更新:', data);
        // 刷新隧道列表
        if (currentPage === 'tunnels') {
            loadPage();
        }
    });
    
    wsClient.connect();
}

// 更新连接状态显示
function updateConnectionStatus(connected) {
    const statusEl = document.querySelector('.ws-status');
    if (statusEl) {
        statusEl.textContent = connected ? '● 实时连接' : '○ 已断开';
        statusEl.style.color = connected ? '#27ae60' : '#e74c3c';
    }
}

// 页面加载时初始化
if (typeof window !== 'undefined') {
    window.addEventListener('DOMContentLoaded', () => {
        initWebSocket();
    });
}

