// API基础URL
const API_BASE = '/api';

// API工具函数
const api = {
    async get(url) {
        const response = await fetch(`${API_BASE}${url}`);
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        return await response.json();
    },
    
    async post(url, data) {
        const response = await fetch(`${API_BASE}${url}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data),
        });
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        return await response.json();
    },
    
    async delete(url) {
        const response = await fetch(`${API_BASE}${url}`, {
            method: 'DELETE',
        });
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        return await response.json();
    },
};

// 路由
const routes = {
    '/': 'dashboard',
    '/clients': 'clients',
    '/tunnels': 'tunnels',
    '/domains': 'domains',
    '/monitor': 'monitor',
};

// 当前页面
let currentPage = 'dashboard';

// 初始化
document.addEventListener('DOMContentLoaded', () => {
    initNavigation();
    loadPage();
    setInterval(loadPage, 30000); // 每30秒刷新
});

// 初始化导航
function initNavigation() {
    const nav = document.querySelector('.navbar nav');
    Object.keys(routes).forEach(path => {
        const a = document.createElement('a');
        a.href = '#';
        a.textContent = getPageTitle(routes[path]);
        a.onclick = (e) => {
            e.preventDefault();
            currentPage = routes[path];
            loadPage();
            updateActiveNav();
        };
        nav.appendChild(a);
    });
    updateActiveNav();
}

function updateActiveNav() {
    document.querySelectorAll('.navbar nav a').forEach((a, i) => {
        if (Object.values(routes)[i] === currentPage) {
            a.classList.add('active');
        } else {
            a.classList.remove('active');
        }
    });
}

function getPageTitle(page) {
    const titles = {
        dashboard: '仪表盘',
        clients: '客户端',
        tunnels: '隧道',
        domains: '域名',
        monitor: '监控',
    };
    return titles[page] || page;
}

// 加载页面
async function loadPage() {
    const app = document.getElementById('app');
    
    try {
        switch (currentPage) {
            case 'dashboard':
                app.innerHTML = await renderDashboard();
                break;
            case 'clients':
                app.innerHTML = await renderClients();
                break;
            case 'tunnels':
                app.innerHTML = await renderTunnels();
                break;
            case 'domains':
                app.innerHTML = await renderDomains();
                break;
            case 'monitor':
                app.innerHTML = await renderMonitor();
                break;
        }
    } catch (error) {
        app.innerHTML = `<div class="card"><p style="color: red;">加载失败: ${error.message}</p></div>`;
    }
}

// 渲染仪表盘
async function renderDashboard() {
    const stats = await api.get('/stats');
    const health = await api.get('/monitor/health');
    
    return `
        <div class="container">
            <div class="stats-grid">
                <div class="stat-card">
                    <h3>连接数</h3>
                    <div class="value">${stats.connections}</div>
                </div>
                <div class="stat-card">
                    <h3>隧道总数</h3>
                    <div class="value">${stats.tunnels}</div>
                </div>
                <div class="stat-card">
                    <h3>活跃隧道</h3>
                    <div class="value">${stats.active_tunnels}</div>
                </div>
                <div class="stat-card">
                    <h3>系统状态</h3>
                    <div class="value" style="font-size: 1.2rem; color: ${health.overall_status === 'ok' ? '#27ae60' : '#e74c3c'}">
                        ${health.overall_status === 'ok' ? '正常' : '警告'}
                    </div>
                </div>
            </div>
            
            <div class="card">
                <h2>系统健康</h2>
                <table>
                    <tr>
                        <th>指标</th>
                        <th>值</th>
                        <th>状态</th>
                    </tr>
                    <tr>
                        <td>CPU使用率</td>
                        <td>${health.cpu?.percent || 0}%</td>
                        <td><span class="status-badge status-${health.cpu?.status || 'unknown'}">${health.cpu?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>内存使用率</td>
                        <td>${health.memory?.percent || 0}%</td>
                        <td><span class="status-badge status-${health.memory?.status || 'unknown'}">${health.memory?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>磁盘使用率</td>
                        <td>${health.disk?.percent || 0}%</td>
                        <td><span class="status-badge status-${health.disk?.status || 'unknown'}">${health.disk?.status || 'unknown'}</span></td>
                    </tr>
                </table>
            </div>
        </div>
    `;
}

// 渲染客户端列表
async function renderClients() {
    const clients = await api.get('/clients');
    
    let tableRows = '';
    clients.forEach(client => {
        tableRows += `
            <tr>
                <td>${client.name || client.id}</td>
                <td>${client.id}</td>
                <td><span class="status-badge status-${client.status}">${client.status}</span></td>
                <td>${client.ip_address || '-'}</td>
                <td>${client.last_seen ? new Date(client.last_seen).toLocaleString() : '-'}</td>
                <td>
                    <button class="btn btn-primary" onclick="viewClient('${client.id}')">查看</button>
                </td>
            </tr>
        `;
    });
    
    return `
        <div class="container">
            <div class="card">
                <h2>客户端列表</h2>
                <button class="btn btn-primary" onclick="createClient()">创建客户端</button>
                <table>
                    <thead>
                        <tr>
                            <th>名称</th>
                            <th>ID</th>
                            <th>状态</th>
                            <th>IP地址</th>
                            <th>最后在线</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${tableRows || '<tr><td colspan="6" style="text-align: center;">暂无客户端</td></tr>'}
                    </tbody>
                </table>
            </div>
        </div>
    `;
}

// 渲染隧道列表
async function renderTunnels() {
    const clients = await api.get('/clients');
    let allTunnels = [];
    
    for (const client of clients) {
        try {
            const tunnels = await api.get(`/clients/${client.id}/tunnels`);
            allTunnels = allTunnels.concat(tunnels.map(t => ({...t, client_name: client.name || client.id})));
        } catch (e) {
            console.error(`Failed to load tunnels for client ${client.id}:`, e);
        }
    }
    
    let tableRows = '';
    allTunnels.forEach(tunnel => {
        tableRows += `
            <tr>
                <td>${tunnel.name}</td>
                <td>${tunnel.client_name}</td>
                <td>${tunnel.tunnel_type}</td>
                <td>${tunnel.local_host}:${tunnel.local_port}</td>
                <td>${tunnel.remote_port || '-'}</td>
                <td><span class="status-badge status-${tunnel.status}">${tunnel.status}</span></td>
                <td>
                    ${tunnel.status === 'active' 
                        ? `<button class="btn btn-warning" onclick="deactivateTunnel('${tunnel.id}')">停用</button>`
                        : `<button class="btn btn-success" onclick="activateTunnel('${tunnel.id}')">激活</button>`
                    }
                    <button class="btn btn-danger" onclick="deleteTunnel('${tunnel.id}')">删除</button>
                </td>
            </tr>
        `;
    });
    
    return `
        <div class="container">
            <div class="card">
                <h2>隧道列表</h2>
                <table>
                    <thead>
                        <tr>
                            <th>名称</th>
                            <th>客户端</th>
                            <th>类型</th>
                            <th>本地地址</th>
                            <th>远程端口</th>
                            <th>状态</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${tableRows || '<tr><td colspan="7" style="text-align: center;">暂无隧道</td></tr>'}
                    </tbody>
                </table>
            </div>
        </div>
    `;
}

// 渲染域名列表
async function renderDomains() {
    const clients = await api.get('/clients');
    let allDomains = [];
    
    for (const client of clients) {
        try {
            const domains = await api.get(`/clients/${client.id}/domains`);
            allDomains = allDomains.concat(domains.map(d => ({...d, client_name: client.name || client.id})));
        } catch (e) {
            console.error(`Failed to load domains for client ${client.id}:`, e);
        }
    }
    
    let tableRows = '';
    allDomains.forEach(domain => {
        tableRows += `
            <tr>
                <td>${domain.domain}</td>
                <td>${domain.client_name}</td>
                <td><span class="status-badge status-${domain.status}">${domain.status}</span></td>
                <td>${domain.ssl_enabled ? '是' : '否'}</td>
                <td>
                    <button class="btn btn-primary" onclick="verifyDomain('${domain.id}')">验证</button>
                    ${domain.ssl_enabled 
                        ? `<button class="btn btn-success" onclick="viewSSLInfo('${domain.id}')">证书信息</button>`
                        : `<button class="btn btn-warning" onclick="requestSSL('${domain.id}')">申请SSL</button>`
                    }
                </td>
            </tr>
        `;
    });
    
    return `
        <div class="container">
            <div class="card">
                <h2>域名列表</h2>
                <table>
                    <thead>
                        <tr>
                            <th>域名</th>
                            <th>客户端</th>
                            <th>状态</th>
                            <th>SSL</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${tableRows || '<tr><td colspan="5" style="text-align: center;">暂无域名</td></tr>'}
                    </tbody>
                </table>
            </div>
        </div>
    `;
}

// 渲染监控页面
async function renderMonitor() {
    const health = await api.get('/monitor/health');
    
    return `
        <div class="container">
            <div class="card">
                <h2>系统监控</h2>
                <p>最后更新: ${new Date(health.timestamp).toLocaleString()}</p>
                <table>
                    <tr>
                        <th>指标</th>
                        <th>值</th>
                        <th>状态</th>
                    </tr>
                    <tr>
                        <td>CPU使用率</td>
                        <td>${health.cpu?.percent || 0}%</td>
                        <td><span class="status-badge status-${health.cpu?.status || 'unknown'}">${health.cpu?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>内存使用</td>
                        <td>${formatBytes(health.memory?.used || 0)} / ${formatBytes(health.memory?.total || 0)} (${health.memory?.percent || 0}%)</td>
                        <td><span class="status-badge status-${health.memory?.status || 'unknown'}">${health.memory?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>磁盘使用</td>
                        <td>${formatBytes(health.disk?.used || 0)} / ${formatBytes(health.disk?.total || 0)} (${health.disk?.percent || 0}%)</td>
                        <td><span class="status-badge status-${health.disk?.status || 'unknown'}">${health.disk?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>连接数</td>
                        <td>${health.connections?.count || 0} / ${health.connections?.max || 0}</td>
                        <td><span class="status-badge status-${health.connections?.status || 'unknown'}">${health.connections?.status || 'unknown'}</span></td>
                    </tr>
                    <tr>
                        <td>隧道</td>
                        <td>活跃: ${health.tunnels?.active || 0} / 总数: ${health.tunnels?.total || 0}</td>
                        <td><span class="status-badge status-${health.tunnels?.status || 'unknown'}">${health.tunnels?.status || 'unknown'}</span></td>
                    </tr>
                </table>
            </div>
        </div>
    `;
}

// 工具函数
function formatBytes(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// 操作函数
async function activateTunnel(tunnelId) {
    try {
        await api.post(`/tunnels/${tunnelId}/activate`);
        alert('隧道已激活');
        loadPage();
    } catch (e) {
        alert('操作失败: ' + e.message);
    }
}

async function deactivateTunnel(tunnelId) {
    try {
        await api.post(`/tunnels/${tunnelId}/deactivate`);
        alert('隧道已停用');
        loadPage();
    } catch (e) {
        alert('操作失败: ' + e.message);
    }
}

async function deleteTunnel(tunnelId) {
    if (!confirm('确定要删除这个隧道吗？')) return;
    try {
        await api.delete(`/tunnels/${tunnelId}`);
        alert('隧道已删除');
        loadPage();
    } catch (e) {
        alert('操作失败: ' + e.message);
    }
}

async function verifyDomain(domainId) {
    try {
        const result = await api.post(`/domains/${domainId}/verify`);
        alert(result.success ? '域名验证成功' : '域名验证失败');
        loadPage();
    } catch (e) {
        alert('操作失败: ' + e.message);
    }
}

async function requestSSL(domainId) {
    try {
        const result = await api.post(`/domains/${domainId}/ssl/request`);
        alert(result.success ? 'SSL证书申请成功' : 'SSL证书申请失败');
        loadPage();
    } catch (e) {
        alert('操作失败: ' + e.message);
    }
}

async function viewSSLInfo(domainId) {
    try {
        const info = await api.get(`/domains/${domainId}/ssl/info`);
        alert(`证书信息:\n域名: ${info.domain}\n证书路径: ${info.cert_path}\n密钥路径: ${info.key_path}`);
    } catch (e) {
        alert('获取证书信息失败: ' + e.message);
    }
}

async function createClient() {
    const name = prompt('请输入客户端名称:');
    if (!name) return;
    
    try {
        const client = await api.post('/clients', { name });
        alert(`客户端创建成功!\nID: ${client.id}\nToken: ${client.token}`);
        loadPage();
    } catch (e) {
        alert('创建失败: ' + e.message);
    }
}

function viewClient(clientId) {
    alert('查看客户端详情功能待实现');
}

// 导出到全局
window.activateTunnel = activateTunnel;
window.deactivateTunnel = deactivateTunnel;
window.deleteTunnel = deleteTunnel;
window.verifyDomain = verifyDomain;
window.requestSSL = requestSSL;
window.viewSSLInfo = viewSSLInfo;
window.createClient = createClient;
window.viewClient = viewClient;

