"""
NPS服务器主程序
"""
import os
import uvicorn
from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from contextlib import asynccontextmanager
from loguru import logger

from common.utils.logger import setup_logger
from .api.routes import router
from .core.gateway.connection_manager import ConnectionManager
from .core.auth.auth_manager import AuthManager
from .core.tunnel.tunnel_manager import TunnelManager
from .core.tunnel.forward_engine import ForwardEngine
from .core.domain.domain_manager import DomainManager
from .core.domain.ssl_manager import SSLManager
from .core.stats.traffic_stats import TrafficStats
from .core.monitor.monitor import SystemMonitor
from .core.quota.quota_manager import QuotaManager
from .core.cache.cache_manager import CacheManager
from .core.logging.log_analyzer import LogAnalyzer
from .core.service_discovery.discovery import ServiceDiscovery
from .core.security.ip_filter import IPFilter
from .core.security.rate_limiter import RateLimiter
from .core.security.anomaly_detector import AnomalyDetector
from .core.monitoring.prometheus_metrics import PrometheusMetrics
from .core.alerting.alert_manager import AlertManager, AlertLevel
from .core.config.config_reloader import ConfigReloader
from .core.backup.backup_manager import BackupManager
from .core.cluster.node_manager import NodeManager
from .core.cluster.session_sync import SessionSync
from .core.api_auth.api_auth import APIAuth
from .core.performance.profiler import PerformanceProfiler, RequestProfiler


# 应用状态
class AppState:
    def __init__(self):
        self.connection_manager: ConnectionManager = None
        self.auth_manager: AuthManager = None
        self.tunnel_manager: TunnelManager = None
        self.domain_manager: DomainManager = None
        self.ssl_manager: SSLManager = None
        self.traffic_stats: TrafficStats = None
        self.system_monitor: SystemMonitor = None
        self.quota_manager: QuotaManager = None
        self.cache_manager: CacheManager = None
        self.log_analyzer: LogAnalyzer = None
        self.service_discovery: ServiceDiscovery = None
        self.broadcaster = None
        self.ip_filter: IPFilter = None
        self.rate_limiter: RateLimiter = None
        self.anomaly_detector: AnomalyDetector = None
        self.prometheus_metrics: PrometheusMetrics = None
        self.alert_manager: AlertManager = None
        self.config_reloader: ConfigReloader = None
        self.backup_manager: BackupManager = None
        self.node_manager: NodeManager = None
        self.session_sync: SessionSync = None
        self.api_auth: APIAuth = None
        self.performance_profiler: PerformanceProfiler = None
        self.request_profiler: RequestProfiler = None


app_state = AppState()


@asynccontextmanager
async def lifespan(app: FastAPI):
    """应用生命周期管理"""
    # 启动时初始化
    logger.info("初始化NPS服务器...")
    
    # 初始化管理器
    secret_key = os.getenv("SECRET_KEY", "your-secret-key-change-in-production")
    app_state.auth_manager = AuthManager(secret_key=secret_key)
    app_state.connection_manager = ConnectionManager()
    app_state.tunnel_manager = TunnelManager()
    
    # 初始化转发引擎
    forward_engine = ForwardEngine(app_state.tunnel_manager)
    app_state.tunnel_manager.set_forward_engine(forward_engine)
    
    # 初始化数据库
    from .db.database import init_db
    try:
        init_db()
        logger.info("数据库初始化成功")
    except Exception as e:
        logger.warning(f"数据库初始化失败（将使用内存模式）: {e}")
    
    # 初始化域名管理
    app_state.domain_manager = DomainManager()
    app_state.ssl_manager = SSLManager()
    
    # 初始化流量统计
    app_state.traffic_stats = TrafficStats()
    
    # 初始化系统监控
    app_state.system_monitor = SystemMonitor(
        connection_manager=app_state.connection_manager,
        tunnel_manager=app_state.tunnel_manager,
    )
    
    # 初始化配额管理
    app_state.quota_manager = QuotaManager()
    
    # 初始化缓存管理器
    app_state.cache_manager = CacheManager(default_ttl=300)
    
    # 初始化日志分析器
    app_state.log_analyzer = LogAnalyzer()
    
    # 初始化服务发现
    app_state.service_discovery = ServiceDiscovery()
    
    # 初始化安全模块
    app_state.ip_filter = IPFilter()
    app_state.rate_limiter = RateLimiter()
    app_state.anomaly_detector = AnomalyDetector()
    
    # 初始化Prometheus指标
    app_state.prometheus_metrics = PrometheusMetrics()
    
    # 初始化告警管理器
    app_state.alert_manager = AlertManager()
    
    # 初始化配置热重载
    app_state.config_reloader = ConfigReloader()
    app_state.config_reloader.load_config()
    app_state.config_reloader.start_watching()
    
    # 初始化备份管理器
    app_state.backup_manager = BackupManager()
    
    # 初始化集群管理
    import uuid
    node_id = os.getenv("NODE_ID", f"node-{uuid.uuid4().hex[:8]}")
    app_state.node_manager = NodeManager(
        node_id=node_id,
        listen_address=os.getenv("HOST", "0.0.0.0"),
        listen_port=int(os.getenv("PORT", "8080"))
    )
    
    # 初始化会话同步
    redis_url = os.getenv("REDIS_URL", "redis://localhost:6379/0")
    app_state.session_sync = SessionSync(redis_url=redis_url)
    asyncio.create_task(app_state.session_sync.connect())
    
    # 初始化API认证
    app_state.api_auth = APIAuth(secret_key=secret_key)
    
    # 初始化性能分析器
    app_state.performance_profiler = PerformanceProfiler()
    app_state.request_profiler = RequestProfiler()
    
    # 启动后台监控任务
    import asyncio
    
    async def background_tasks():
        """后台任务"""
        tasks = [
            app_state.system_monitor.start_monitoring(),
            cache_cleanup_task(app_state.cache_manager),
            metrics_update_task(app_state.prometheus_metrics, app_state.system_monitor),
        ]
        
        # 如果启用集群，启动心跳
        if app_state.node_manager:
            cluster_enabled = os.getenv("CLUSTER_ENABLED", "false").lower() == "true"
            if cluster_enabled:
                app_state.node_manager.enable_cluster(True)
                tasks.append(app_state.node_manager.start_heartbeat_loop())
        
        await asyncio.gather(*tasks, return_exceptions=True)
    
    async def cache_cleanup_task(cache_manager):
        """缓存清理任务"""
        while True:
            await asyncio.sleep(300)  # 每5分钟清理一次
            cache_manager.cleanup_expired()
    
    async def metrics_update_task(metrics, monitor):
        """指标更新任务"""
        import psutil
        while True:
            try:
                health = await monitor.check_system_health()
                if health:
                    metrics.set_system_metrics(
                        cpu_percent=health.get('cpu', {}).get('percent', 0),
                        memory_used=health.get('memory', {}).get('used', 0),
                        memory_total=health.get('memory', {}).get('total', 0),
                        memory_available=health.get('memory', {}).get('total', 0) - health.get('memory', {}).get('used', 0),
                    )
            except Exception as e:
                logger.error(f"更新指标失败: {e}")
            await asyncio.sleep(60)  # 每分钟更新一次
    
    monitor_task = asyncio.create_task(background_tasks())
    
    # 等待监控任务完成（实际上会一直运行）
    try:
        await monitor_task
    except asyncio.CancelledError:
        pass
    
    logger.info("NPS服务器启动完成")
    
    yield
    
    # 关闭时清理
    logger.info("关闭NPS服务器...")
    
    # 停止监控
    if app_state.system_monitor:
        app_state.system_monitor.stop_monitoring()
    
    # 停止配置监控
    if app_state.config_reloader:
        app_state.config_reloader.stop_watching()
    
    # 断开会话同步
    if app_state.session_sync:
        await app_state.session_sync.disconnect()


# 创建FastAPI应用
app = FastAPI(
    title="NPS Server",
    description="内网穿透服务器",
    version="1.0.0",
    lifespan=lifespan,
)

# CORS中间件
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# 安全中间件
@app.middleware("http")
async def security_middleware(request: Request, call_next):
    """安全中间件：IP过滤和速率限制"""
    # IP过滤
    client_ip = request.client.host if request.client else "unknown"
    if app_state.ip_filter and not app_state.ip_filter.is_allowed(client_ip):
        from fastapi.responses import Response
        return Response(
            content="IP address not allowed",
            status_code=403
        )
    
    # 速率限制
    if app_state.rate_limiter:
        rate_limit_key = f"{client_ip}:{request.url.path}"
        allowed, limit_info = app_state.rate_limiter.is_allowed(rate_limit_key)
        
        if not allowed:
            from fastapi.responses import Response
            import time
            return Response(
                content="Rate limit exceeded",
                status_code=429,
                headers={
                    "X-RateLimit-Limit": str(limit_info.get('limit', 0)),
                    "X-RateLimit-Remaining": "0",
                    "Retry-After": str(int(limit_info.get('reset_at', 0) - time.time())),
                }
            )
    
    # 记录请求开始时间
    import time
    start_time = time.time()
    
    # 执行请求
    response = await call_next(request)
    
    # 记录请求指标
    if app_state.prometheus_metrics:
        duration = time.time() - start_time
        app_state.prometheus_metrics.record_http_request(
            method=request.method,
            endpoint=str(request.url.path),
            status_code=response.status_code,
            duration=duration
        )
    
    return response

# 注册路由
app.include_router(router, prefix="/api", tags=["api"])

# 静态文件服务
try:
    app.mount("/static", StaticFiles(directory="server/web/static"), name="static")
except Exception as e:
    logger.warning(f"无法加载静态文件: {e}")

# 根路径重定向到管理面板
@app.get("/admin")
async def admin_panel():
    """管理面板"""
    from fastapi.responses import FileResponse
    import os
    static_path = os.path.join(os.path.dirname(__file__), "web/static/index.html")
    if os.path.exists(static_path):
        return FileResponse(static_path)
    return {"message": "管理面板未找到，请检查静态文件路径"}


@app.get("/")
async def root():
    """根路径"""
    return {
        "name": "NPS Server",
        "version": "1.0.0",
        "status": "running",
        "api_docs": "/docs",
        "admin_panel": "/admin",
    }


def main():
    """主函数"""
    # 配置日志
    setup_logger(level="INFO", log_file="logs/server.log")
    
    # 获取配置
    host = os.getenv("HOST", "0.0.0.0")
    port = int(os.getenv("PORT", "8080"))
    
    logger.info(f"启动服务器: http://{host}:{port}")
    
    # 启动服务器
    uvicorn.run(
        "server.main:app",
        host=host,
        port=port,
        reload=False,
        log_level="info",
    )


if __name__ == "__main__":
    main()

