"""
数据访问层（Repository模式）
"""
from typing import List, Optional
from sqlalchemy.orm import Session
from sqlalchemy import and_, func
from datetime import datetime, timedelta

from .models import (
    ClientModel, TunnelModel, DomainModel, SessionModel, TrafficStatsModel
)
from common.models.client import ClientStatus
from common.models.tunnel import TunnelType, TunnelStatus
from common.models.domain import DomainStatus
from common.models.session import SessionStatus


class ClientRepository:
    """客户端数据访问"""
    
    @staticmethod
    def create(db: Session, name: str, token: str, **kwargs) -> ClientModel:
        """创建客户端"""
        client = ClientModel(
            name=name,
            token=token,
            **kwargs
        )
        db.add(client)
        db.commit()
        db.refresh(client)
        return client
    
    @staticmethod
    def get_by_id(db: Session, client_id: str) -> Optional[ClientModel]:
        """根据ID获取客户端"""
        return db.query(ClientModel).filter(ClientModel.id == client_id).first()
    
    @staticmethod
    def get_by_token(db: Session, token: str) -> Optional[ClientModel]:
        """根据Token获取客户端"""
        return db.query(ClientModel).filter(ClientModel.token == token).first()
    
    @staticmethod
    def get_all(db: Session) -> List[ClientModel]:
        """获取所有客户端"""
        return db.query(ClientModel).all()
    
    @staticmethod
    def update_status(db: Session, client_id: str, status: ClientStatus, ip_address: str = None):
        """更新客户端状态"""
        client = ClientRepository.get_by_id(db, client_id)
        if client:
            client.status = status
            client.last_seen = datetime.utcnow()
            if ip_address:
                client.ip_address = ip_address
            db.commit()
    
    @staticmethod
    def delete(db: Session, client_id: str):
        """删除客户端"""
        client = ClientRepository.get_by_id(db, client_id)
        if client:
            db.delete(client)
            db.commit()


class TunnelRepository:
    """隧道数据访问"""
    
    @staticmethod
    def create(db: Session, client_id: str, name: str, tunnel_type: TunnelType,
               local_host: str, local_port: int, **kwargs) -> TunnelModel:
        """创建隧道"""
        tunnel = TunnelModel(
            client_id=client_id,
            name=name,
            tunnel_type=tunnel_type,
            local_host=local_host,
            local_port=local_port,
            **kwargs
        )
        db.add(tunnel)
        db.commit()
        db.refresh(tunnel)
        return tunnel
    
    @staticmethod
    def get_by_id(db: Session, tunnel_id: str) -> Optional[TunnelModel]:
        """根据ID获取隧道"""
        return db.query(TunnelModel).filter(TunnelModel.id == tunnel_id).first()
    
    @staticmethod
    def get_by_client(db: Session, client_id: str) -> List[TunnelModel]:
        """获取客户端的所有隧道"""
        return db.query(TunnelModel).filter(TunnelModel.client_id == client_id).all()
    
    @staticmethod
    def get_by_port(db: Session, port: int) -> Optional[TunnelModel]:
        """根据端口获取隧道"""
        return db.query(TunnelModel).filter(TunnelModel.remote_port == port).first()
    
    @staticmethod
    def update_status(db: Session, tunnel_id: str, status: TunnelStatus):
        """更新隧道状态"""
        tunnel = TunnelRepository.get_by_id(db, tunnel_id)
        if tunnel:
            tunnel.status = status
            tunnel.updated_at = datetime.utcnow()
            db.commit()
    
    @staticmethod
    def delete(db: Session, tunnel_id: str):
        """删除隧道"""
        tunnel = TunnelRepository.get_by_id(db, tunnel_id)
        if tunnel:
            db.delete(tunnel)
            db.commit()


class DomainRepository:
    """域名数据访问"""
    
    @staticmethod
    def create(db: Session, client_id: str, domain: str, **kwargs) -> DomainModel:
        """创建域名"""
        domain_obj = DomainModel(
            client_id=client_id,
            domain=domain,
            **kwargs
        )
        db.add(domain_obj)
        db.commit()
        db.refresh(domain_obj)
        return domain_obj
    
    @staticmethod
    def get_by_id(db: Session, domain_id: str) -> Optional[DomainModel]:
        """根据ID获取域名"""
        return db.query(DomainModel).filter(DomainModel.id == domain_id).first()
    
    @staticmethod
    def get_by_name(db: Session, domain: str) -> Optional[DomainModel]:
        """根据域名名称获取"""
        return db.query(DomainModel).filter(DomainModel.domain == domain).first()
    
    @staticmethod
    def get_by_client(db: Session, client_id: str) -> List[DomainModel]:
        """获取客户端的所有域名"""
        return db.query(DomainModel).filter(DomainModel.client_id == client_id).all()
    
    @staticmethod
    def update_status(db: Session, domain_id: str, status: DomainStatus):
        """更新域名状态"""
        domain = DomainRepository.get_by_id(db, domain_id)
        if domain:
            domain.status = status
            domain.updated_at = datetime.utcnow()
            db.commit()
    
    @staticmethod
    def delete(db: Session, domain_id: str):
        """删除域名"""
        domain = DomainRepository.get_by_id(db, domain_id)
        if domain:
            db.delete(domain)
            db.commit()


class TrafficStatsRepository:
    """流量统计数据访问"""
    
    @staticmethod
    def record(db: Session, client_id: str = None, tunnel_id: str = None,
               bytes_sent: int = 0, bytes_received: int = 0):
        """记录流量"""
        stats = TrafficStatsModel(
            client_id=client_id,
            tunnel_id=tunnel_id,
            bytes_sent=bytes_sent,
            bytes_received=bytes_received,
        )
        db.add(stats)
        db.commit()
    
    @staticmethod
    def get_client_stats(db: Session, client_id: str, hours: int = 24) -> dict:
        """获取客户端统计"""
        cutoff_time = datetime.utcnow() - timedelta(hours=hours)
        
        # 总流量
        total = db.query(
            func.sum(TrafficStatsModel.bytes_sent).label('sent'),
            func.sum(TrafficStatsModel.bytes_received).label('received')
        ).filter(
            and_(
                TrafficStatsModel.client_id == client_id,
                TrafficStatsModel.timestamp >= cutoff_time
            )
        ).first()
        
        return {
            'total_bytes_sent': total.sent or 0,
            'total_bytes_received': total.received or 0,
            'total_bytes': (total.sent or 0) + (total.received or 0),
        }
    
    @staticmethod
    def get_tunnel_stats(db: Session, tunnel_id: str, hours: int = 24) -> dict:
        """获取隧道统计"""
        cutoff_time = datetime.utcnow() - timedelta(hours=hours)
        
        total = db.query(
            func.sum(TrafficStatsModel.bytes_sent).label('sent'),
            func.sum(TrafficStatsModel.bytes_received).label('received')
        ).filter(
            and_(
                TrafficStatsModel.tunnel_id == tunnel_id,
                TrafficStatsModel.timestamp >= cutoff_time
            )
        ).first()
        
        return {
            'total_bytes_sent': total.sent or 0,
            'total_bytes_received': total.received or 0,
            'total_bytes': (total.sent or 0) + (total.received or 0),
        }

