"""
数据库模型
"""
from sqlalchemy import Column, String, Integer, Boolean, DateTime, ForeignKey, Text, Enum as SQLEnum
from sqlalchemy.orm import relationship
from datetime import datetime
import uuid

from .database import Base
from common.models.client import ClientStatus
from common.models.tunnel import TunnelType, TunnelStatus
from common.models.domain import DomainStatus
from common.models.session import SessionStatus


def generate_id():
    """生成UUID"""
    return str(uuid.uuid4())


class ClientModel(Base):
    """客户端模型"""
    __tablename__ = "clients"
    
    id = Column(String, primary_key=True, default=generate_id)
    name = Column(String, nullable=False)
    token = Column(String, nullable=False, unique=True, index=True)
    status = Column(SQLEnum(ClientStatus), default=ClientStatus.OFFLINE)
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    last_seen = Column(DateTime, nullable=True)
    ip_address = Column(String, nullable=True)
    version = Column(String, nullable=True)
    description = Column(Text, nullable=True)
    
    # 关系
    tunnels = relationship("TunnelModel", back_populates="client", cascade="all, delete-orphan")
    domains = relationship("DomainModel", back_populates="client", cascade="all, delete-orphan")
    sessions = relationship("SessionModel", back_populates="client", cascade="all, delete-orphan")
    
    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'status': self.status.value if self.status else None,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'last_seen': self.last_seen.isoformat() if self.last_seen else None,
            'ip_address': self.ip_address,
            'version': self.version,
            'description': self.description,
        }


class TunnelModel(Base):
    """隧道模型"""
    __tablename__ = "tunnels"
    
    id = Column(String, primary_key=True, default=generate_id)
    client_id = Column(String, ForeignKey("clients.id"), nullable=False, index=True)
    name = Column(String, nullable=False)
    tunnel_type = Column(SQLEnum(TunnelType), nullable=False)
    local_host = Column(String, nullable=False)
    local_port = Column(Integer, nullable=False)
    remote_port = Column(Integer, nullable=True)
    domain = Column(String, nullable=True)
    status = Column(SQLEnum(TunnelStatus), default=TunnelStatus.INACTIVE)
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    description = Column(Text, nullable=True)
    
    # 关系
    client = relationship("ClientModel", back_populates="tunnels")
    
    def to_dict(self):
        return {
            'id': self.id,
            'client_id': self.client_id,
            'name': self.name,
            'tunnel_type': self.tunnel_type.value if self.tunnel_type else None,
            'local_host': self.local_host,
            'local_port': self.local_port,
            'remote_port': self.remote_port,
            'domain': self.domain,
            'status': self.status.value if self.status else None,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'description': self.description,
        }


class DomainModel(Base):
    """域名模型"""
    __tablename__ = "domains"
    
    id = Column(String, primary_key=True, default=generate_id)
    domain = Column(String, nullable=False, unique=True, index=True)
    client_id = Column(String, ForeignKey("clients.id"), nullable=False, index=True)
    tunnel_id = Column(String, ForeignKey("tunnels.id"), nullable=True)
    ssl_enabled = Column(Boolean, default=False)
    ssl_cert_path = Column(String, nullable=True)
    ssl_key_path = Column(String, nullable=True)
    status = Column(SQLEnum(DomainStatus), default=DomainStatus.PENDING)
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # 关系
    client = relationship("ClientModel", back_populates="domains")
    
    def to_dict(self):
        return {
            'id': self.id,
            'domain': self.domain,
            'client_id': self.client_id,
            'tunnel_id': self.tunnel_id,
            'ssl_enabled': self.ssl_enabled,
            'status': self.status.value if self.status else None,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
        }


class SessionModel(Base):
    """会话模型"""
    __tablename__ = "sessions"
    
    id = Column(String, primary_key=True, default=generate_id)
    client_id = Column(String, ForeignKey("clients.id"), nullable=False, index=True)
    status = Column(SQLEnum(SessionStatus), default=SessionStatus.CONNECTING)
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    remote_addr = Column(String, nullable=True)
    protocol = Column(String, nullable=True)
    
    # 关系
    client = relationship("ClientModel", back_populates="sessions")
    
    def to_dict(self):
        return {
            'id': self.id,
            'client_id': self.client_id,
            'status': self.status.value if self.status else None,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'updated_at': self.updated_at.isoformat() if self.updated_at else None,
            'remote_addr': self.remote_addr,
            'protocol': self.protocol,
        }


class TrafficStatsModel(Base):
    """流量统计模型"""
    __tablename__ = "traffic_stats"
    
    id = Column(String, primary_key=True, default=generate_id)
    client_id = Column(String, ForeignKey("clients.id"), nullable=True, index=True)
    tunnel_id = Column(String, ForeignKey("tunnels.id"), nullable=True, index=True)
    bytes_sent = Column(Integer, default=0)
    bytes_received = Column(Integer, default=0)
    timestamp = Column(DateTime, default=datetime.utcnow, index=True)
    
    def to_dict(self):
        return {
            'id': self.id,
            'client_id': self.client_id,
            'tunnel_id': self.tunnel_id,
            'bytes_sent': self.bytes_sent,
            'bytes_received': self.bytes_received,
            'timestamp': self.timestamp.isoformat() if self.timestamp else None,
        }

