"""
UDP代理实现
"""
import asyncio
import socket
from typing import Dict, Optional, Tuple
from loguru import logger
from collections import defaultdict

from common.models.tunnel import Tunnel, TunnelStatus


class UDPProxy:
    """UDP代理"""
    
    def __init__(self):
        self._udp_sockets: Dict[str, socket.socket] = {}
        self._client_sessions: Dict[Tuple[str, int], str] = {}  # (client_addr, client_port) -> tunnel_id
        self._tunnel_sessions: Dict[str, Dict[Tuple[str, int], socket.socket]] = {}  # tunnel_id -> {(client_addr, port): socket}
    
    async def start_udp_proxy(self, tunnel: Tunnel, local_host: str, local_port: int):
        """启动UDP代理"""
        tunnel_id = tunnel.id
        
        if tunnel_id in self._udp_sockets:
            logger.warning(f"UDP代理已启动: {tunnel_id}")
            return
        
        try:
            # 创建UDP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            sock.bind((local_host, local_port))
            sock.setblocking(False)
            
            self._udp_sockets[tunnel_id] = sock
            self._tunnel_sessions[tunnel_id] = {}
            
            logger.info(f"UDP代理启动: {tunnel_id} -> {tunnel.local_host}:{tunnel.local_port}")
            
            # 启动接收循环
            asyncio.create_task(self._udp_receive_loop(tunnel_id, sock, tunnel))
            
        except Exception as e:
            logger.error(f"启动UDP代理失败: {e}")
            if tunnel_id in self._udp_sockets:
                del self._udp_sockets[tunnel_id]
    
    async def stop_udp_proxy(self, tunnel_id: str):
        """停止UDP代理"""
        if tunnel_id in self._udp_sockets:
            sock = self._udp_sockets[tunnel_id]
            sock.close()
            del self._udp_sockets[tunnel_id]
        
        if tunnel_id in self._tunnel_sessions:
            for session_sock in self._tunnel_sessions[tunnel_id].values():
                try:
                    session_sock.close()
                except:
                    pass
            del self._tunnel_sessions[tunnel_id]
        
        logger.info(f"UDP代理已停止: {tunnel_id}")
    
    async def _udp_receive_loop(self, tunnel_id: str, sock: socket.socket, tunnel: Tunnel):
        """UDP接收循环"""
        loop = asyncio.get_event_loop()
        
        try:
            while tunnel_id in self._udp_sockets:
                try:
                    # 接收来自客户端的数据
                    data, client_addr = await loop.sock_recvfrom(sock, 65535)
                    
                    if not data:
                        continue
                    
                    # 获取或创建到本地服务的连接
                    local_sock = await self._get_local_socket(tunnel_id, client_addr, tunnel)
                    
                    if local_sock:
                        # 转发数据到本地服务
                        await loop.sock_sendto(local_sock, data, (tunnel.local_host, tunnel.local_port))
                    
                except asyncio.CancelledError:
                    break
                except Exception as e:
                    logger.error(f"UDP接收循环错误: {e}")
                    await asyncio.sleep(0.1)
                    
        except Exception as e:
            logger.error(f"UDP接收循环异常: {e}")
        finally:
            if tunnel_id in self._udp_sockets:
                try:
                    self._udp_sockets[tunnel_id].close()
                except:
                    pass
    
    async def _get_local_socket(self, tunnel_id: str, client_addr: Tuple[str, int], tunnel: Tunnel) -> Optional[socket.socket]:
        """获取或创建到本地服务的socket"""
        if tunnel_id not in self._tunnel_sessions:
            return None
        
        if client_addr in self._tunnel_sessions[tunnel_id]:
            return self._tunnel_sessions[tunnel_id][client_addr]
        
        try:
            # 创建新的UDP socket用于与本地服务通信
            local_sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            local_sock.setblocking(False)
            
            self._tunnel_sessions[tunnel_id][client_addr] = local_sock
            
            # 启动接收本地服务响应的循环
            loop = asyncio.get_event_loop()
            asyncio.create_task(self._local_response_loop(tunnel_id, client_addr, local_sock, tunnel))
            
            return local_sock
            
        except Exception as e:
            logger.error(f"创建本地socket失败: {e}")
            return None
    
    async def _local_response_loop(self, tunnel_id: str, client_addr: Tuple[str, int], 
                                  local_sock: socket.socket, tunnel: Tunnel):
        """接收本地服务响应的循环"""
        loop = asyncio.get_event_loop()
        
        try:
            while tunnel_id in self._tunnel_sessions and client_addr in self._tunnel_sessions[tunnel_id]:
                try:
                    # 接收来自本地服务的数据
                    data, _ = await loop.sock_recvfrom(local_sock, 65535)
                    
                    if not data:
                        continue
                    
                    # 转发数据回客户端
                    if tunnel_id in self._udp_sockets:
                        server_sock = self._udp_sockets[tunnel_id]
                        await loop.sock_sendto(server_sock, data, client_addr)
                    
                except asyncio.CancelledError:
                    break
                except Exception as e:
                    logger.debug(f"本地响应循环错误: {e}")
                    break
                    
        except Exception as e:
            logger.error(f"本地响应循环异常: {e}")
        finally:
            # 清理
            if tunnel_id in self._tunnel_sessions and client_addr in self._tunnel_sessions[tunnel_id]:
                try:
                    self._tunnel_sessions[tunnel_id][client_addr].close()
                    del self._tunnel_sessions[tunnel_id][client_addr]
                except:
                    pass

