"""
协议处理器 - 支持更多协议类型
"""
from typing import Dict, Optional
from enum import Enum
from loguru import logger
import asyncio


class ProtocolType(str, Enum):
    """协议类型"""
    TCP = "tcp"
    UDP = "udp"
    HTTP = "http"
    HTTPS = "https"
    WEBSOCKET = "websocket"
    WEBSOCKET_SECURE = "wss"
    SOCKS5 = "socks5"
    FTP = "ftp"
    SSH = "ssh"
    RDP = "rdp"
    VNC = "vnc"


class ProtocolHandler:
    """协议处理器基类"""
    
    def __init__(self, protocol_type: ProtocolType):
        self.protocol_type = protocol_type
    
    async def handle_connection(self, connection, tunnel_config: Dict):
        """处理连接"""
        raise NotImplementedError
    
    async def handle_data(self, data: bytes, connection, tunnel_config: Dict):
        """处理数据"""
        raise NotImplementedError


class SOCKS5Handler(ProtocolHandler):
    """SOCKS5协议处理器"""
    
    def __init__(self):
        super().__init__(ProtocolType.SOCKS5)
    
    async def handle_connection(self, connection, tunnel_config: Dict):
        """处理SOCKS5连接"""
        logger.info(f"处理SOCKS5连接: {tunnel_config}")
        # SOCKS5协议实现
        pass
    
    async def handle_data(self, data: bytes, connection, tunnel_config: Dict):
        """处理SOCKS5数据"""
        # SOCKS5数据处理
        pass


class FTPHandler(ProtocolHandler):
    """FTP协议处理器"""
    
    def __init__(self):
        super().__init__(ProtocolType.FTP)
    
    async def handle_connection(self, connection, tunnel_config: Dict):
        """处理FTP连接"""
        logger.info(f"处理FTP连接: {tunnel_config}")
        # FTP协议实现
        pass
    
    async def handle_data(self, data: bytes, connection, tunnel_config: Dict):
        """处理FTP数据"""
        # FTP数据处理
        pass


class SSHHandler(ProtocolHandler):
    """SSH协议处理器"""
    
    def __init__(self):
        super().__init__(ProtocolType.SSH)
    
    async def handle_connection(self, connection, tunnel_config: Dict):
        """处理SSH连接"""
        logger.info(f"处理SSH连接: {tunnel_config}")
        # SSH协议实现
        pass
    
    async def handle_data(self, data: bytes, connection, tunnel_config: Dict):
        """处理SSH数据"""
        # SSH数据处理
        pass


class ProtocolHandlerFactory:
    """协议处理器工厂"""
    
    _handlers: Dict[ProtocolType, ProtocolHandler] = {}
    
    @classmethod
    def register_handler(cls, protocol_type: ProtocolType, handler: ProtocolHandler):
        """注册协议处理器"""
        cls._handlers[protocol_type] = handler
        logger.info(f"注册协议处理器: {protocol_type.value}")
    
    @classmethod
    def get_handler(cls, protocol_type: ProtocolType) -> Optional[ProtocolHandler]:
        """获取协议处理器"""
        return cls._handlers.get(protocol_type)
    
    @classmethod
    def get_supported_protocols(cls) -> List[ProtocolType]:
        """获取支持的协议列表"""
        return list(cls._handlers.keys())


# 注册默认处理器
ProtocolHandlerFactory.register_handler(ProtocolType.SOCKS5, SOCKS5Handler())
ProtocolHandlerFactory.register_handler(ProtocolType.FTP, FTPHandler())
ProtocolHandlerFactory.register_handler(ProtocolType.SSH, SSHHandler())

