"""
流量统计
"""
import time
from typing import Dict, Optional
from datetime import datetime, timedelta
from collections import defaultdict
from loguru import logger


class TrafficStats:
    """流量统计"""
    
    def __init__(self):
        self._client_stats: Dict[str, 'ClientStats'] = {}
        self._tunnel_stats: Dict[str, 'TunnelStats'] = {}
    
    def record_traffic(self, client_id: str, tunnel_id: str, bytes_sent: int, bytes_received: int):
        """记录流量"""
        # 客户端统计
        if client_id not in self._client_stats:
            self._client_stats[client_id] = ClientStats(client_id)
        self._client_stats[client_id].add_traffic(bytes_sent, bytes_received)
        
        # 隧道统计
        if tunnel_id not in self._tunnel_stats:
            self._tunnel_stats[tunnel_id] = TunnelStats(tunnel_id)
        self._tunnel_stats[tunnel_id].add_traffic(bytes_sent, bytes_received)
    
    def get_client_stats(self, client_id: str, hours: int = 24) -> Optional[dict]:
        """获取客户端统计"""
        if client_id not in self._client_stats:
            return None
        return self._client_stats[client_id].get_stats(hours)
    
    def get_tunnel_stats(self, tunnel_id: str, hours: int = 24) -> Optional[dict]:
        """获取隧道统计"""
        if tunnel_id not in self._tunnel_stats:
            return None
        return self._tunnel_stats[tunnel_id].get_stats(hours)
    
    def get_all_stats(self) -> dict:
        """获取所有统计"""
        return {
            'clients': len(self._client_stats),
            'tunnels': len(self._tunnel_stats),
            'total_bytes_sent': sum(s.total_bytes_sent for s in self._client_stats.values()),
            'total_bytes_received': sum(s.total_bytes_received for s in self._client_stats.values()),
        }


class ClientStats:
    """客户端统计"""
    
    def __init__(self, client_id: str):
        self.client_id = client_id
        self.total_bytes_sent = 0
        self.total_bytes_received = 0
        self._hourly_stats: Dict[int, dict] = defaultdict(lambda: {'sent': 0, 'received': 0})
        self.last_update = datetime.now()
    
    def add_traffic(self, bytes_sent: int, bytes_received: int):
        """添加流量"""
        self.total_bytes_sent += bytes_sent
        self.total_bytes_received += bytes_received
        
        # 按小时统计
        hour_key = datetime.now().replace(minute=0, second=0, microsecond=0)
        hour_timestamp = int(hour_key.timestamp())
        self._hourly_stats[hour_timestamp]['sent'] += bytes_sent
        self._hourly_stats[hour_timestamp]['received'] += bytes_received
        
        self.last_update = datetime.now()
    
    def get_stats(self, hours: int = 24) -> dict:
        """获取统计信息"""
        cutoff_time = datetime.now() - timedelta(hours=hours)
        cutoff_timestamp = int(cutoff_time.replace(minute=0, second=0, microsecond=0).timestamp())
        
        hourly_data = []
        for timestamp, stats in sorted(self._hourly_stats.items()):
            if timestamp >= cutoff_timestamp:
                hourly_data.append({
                    'time': datetime.fromtimestamp(timestamp).isoformat(),
                    'bytes_sent': stats['sent'],
                    'bytes_received': stats['received'],
                })
        
        return {
            'client_id': self.client_id,
            'total_bytes_sent': self.total_bytes_sent,
            'total_bytes_received': self.total_bytes_received,
            'total_bytes': self.total_bytes_sent + self.total_bytes_received,
            'hourly_stats': hourly_data,
            'last_update': self.last_update.isoformat(),
        }


class TunnelStats:
    """隧道统计"""
    
    def __init__(self, tunnel_id: str):
        self.tunnel_id = tunnel_id
        self.total_bytes_sent = 0
        self.total_bytes_received = 0
        self.connection_count = 0
        self._hourly_stats: Dict[int, dict] = defaultdict(lambda: {'sent': 0, 'received': 0, 'connections': 0})
        self.last_update = datetime.now()
    
    def add_traffic(self, bytes_sent: int, bytes_received: int):
        """添加流量"""
        self.total_bytes_sent += bytes_sent
        self.total_bytes_received += bytes_received
        
        hour_key = datetime.now().replace(minute=0, second=0, microsecond=0)
        hour_timestamp = int(hour_key.timestamp())
        self._hourly_stats[hour_timestamp]['sent'] += bytes_sent
        self._hourly_stats[hour_timestamp]['received'] += bytes_received
    
    def add_connection(self):
        """添加连接"""
        self.connection_count += 1
        hour_key = datetime.now().replace(minute=0, second=0, microsecond=0)
        hour_timestamp = int(hour_key.timestamp())
        self._hourly_stats[hour_timestamp]['connections'] += 1
    
    def get_stats(self, hours: int = 24) -> dict:
        """获取统计信息"""
        cutoff_time = datetime.now() - timedelta(hours=hours)
        cutoff_timestamp = int(cutoff_time.replace(minute=0, second=0, microsecond=0).timestamp())
        
        hourly_data = []
        for timestamp, stats in sorted(self._hourly_stats.items()):
            if timestamp >= cutoff_timestamp:
                hourly_data.append({
                    'time': datetime.fromtimestamp(timestamp).isoformat(),
                    'bytes_sent': stats['sent'],
                    'bytes_received': stats['received'],
                    'connections': stats['connections'],
                })
        
        return {
            'tunnel_id': self.tunnel_id,
            'total_bytes_sent': self.total_bytes_sent,
            'total_bytes_received': self.total_bytes_received,
            'total_bytes': self.total_bytes_sent + self.total_bytes_received,
            'connection_count': self.connection_count,
            'hourly_stats': hourly_data,
            'last_update': self.last_update.isoformat(),
        }

