"""
本地服务发现
"""
import socket
import asyncio
from typing import List, Dict, Optional
from dataclasses import dataclass
from loguru import logger


@dataclass
class ServiceInfo:
    """服务信息"""
    name: str
    host: str
    port: int
    protocol: str  # tcp, udp, http
    status: str = "unknown"  # unknown, active, inactive
    description: str = ""


class ServiceDiscovery:
    """服务发现"""
    
    def __init__(self):
        self._common_ports = {
            # Web服务
            80: ("HTTP", "tcp"),
            443: ("HTTPS", "tcp"),
            8080: ("HTTP-Alt", "tcp"),
            8443: ("HTTPS-Alt", "tcp"),
            # 数据库
            3306: ("MySQL", "tcp"),
            5432: ("PostgreSQL", "tcp"),
            6379: ("Redis", "tcp"),
            27017: ("MongoDB", "tcp"),
            # SSH
            22: ("SSH", "tcp"),
            # FTP
            21: ("FTP", "tcp"),
            # 其他
            3389: ("RDP", "tcp"),
            5900: ("VNC", "tcp"),
        }
    
    async def scan_local_services(self, host: str = "127.0.0.1", ports: Optional[List[int]] = None) -> List[ServiceInfo]:
        """扫描本地服务"""
        if ports is None:
            # 扫描常见端口
            ports = list(self._common_ports.keys())
        
        services = []
        tasks = [self._check_port(host, port) for port in ports]
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        for i, result in enumerate(results):
            if isinstance(result, ServiceInfo):
                services.append(result)
            elif isinstance(result, Exception):
                logger.debug(f"检查端口 {ports[i]} 失败: {result}")
        
        return services
    
    async def _check_port(self, host: str, port: int) -> Optional[ServiceInfo]:
        """检查端口是否开放"""
        try:
            # TCP连接检查
            reader, writer = await asyncio.wait_for(
                asyncio.open_connection(host, port),
                timeout=1.0
            )
            writer.close()
            await writer.wait_closed()
            
            # 端口开放，尝试识别服务
            service_name, protocol = self._common_ports.get(port, (f"Port-{port}", "tcp"))
            
            return ServiceInfo(
                name=service_name,
                host=host,
                port=port,
                protocol=protocol,
                status="active",
            )
            
        except asyncio.TimeoutError:
            return None
        except ConnectionRefusedError:
            return None
        except Exception as e:
            logger.debug(f"检查端口 {host}:{port} 异常: {e}")
            return None
    
    async def detect_service_type(self, host: str, port: int) -> Optional[str]:
        """检测服务类型"""
        try:
            reader, writer = await asyncio.wait_for(
                asyncio.open_connection(host, port),
                timeout=2.0
            )
            
            # 尝试读取banner
            try:
                banner = await asyncio.wait_for(reader.read(1024), timeout=1.0)
                banner_str = banner.decode('utf-8', errors='ignore').strip()
                
                # 简单的服务识别
                if b'HTTP' in banner or b'http' in banner:
                    return 'http'
                elif b'SSH' in banner:
                    return 'ssh'
                elif b'FTP' in banner:
                    return 'ftp'
                elif b'MySQL' in banner or b'mysql' in banner:
                    return 'mysql'
                elif b'PostgreSQL' in banner:
                    return 'postgresql'
                    
            except asyncio.TimeoutError:
                pass
            
            writer.close()
            await writer.wait_closed()
            
            # 根据端口猜测
            service_name, _ = self._common_ports.get(port, ("unknown", "tcp"))
            return service_name.lower()
            
        except Exception as e:
            logger.debug(f"检测服务类型失败 {host}:{port}: {e}")
            return None
    
    def get_common_ports(self) -> Dict[int, tuple]:
        """获取常见端口列表"""
        return self._common_ports.copy()

