"""
HTTPS管理器 - 自动配置和启用HTTPS
"""
import os
import ssl
import subprocess
from pathlib import Path
from typing import Optional, Dict
from loguru import logger
from .ssl_manager import SSLManager


class HTTPSManager:
    """HTTPS管理器"""
    
    def __init__(self, cert_dir: str = "certs"):
        self.cert_dir = Path(cert_dir)
        self.cert_dir.mkdir(exist_ok=True, parents=True)
        self.ssl_manager = SSLManager()
        self._enabled = False
        self._cert_file: Optional[Path] = None
        self._key_file: Optional[Path] = None
    
    def enable_https(self, domain: str, email: str = None, auto_renew: bool = True) -> bool:
        """启用HTTPS"""
        try:
            logger.info(f"开始配置HTTPS for {domain}")
            
            # 检查是否已有证书
            cert_file = self.cert_dir / f"{domain}.crt"
            key_file = self.cert_dir / f"{domain}.key"
            
            if cert_file.exists() and key_file.exists():
                logger.info(f"发现现有证书: {cert_file}")
                self._cert_file = cert_file
                self._key_file = key_file
                self._enabled = True
                return True
            
            # 申请新证书
            if email:
                logger.info(f"使用ACME申请证书 for {domain}")
                cert_data = self.ssl_manager.request_certificate(domain, email)
                
                if cert_data:
                    # 保存证书
                    cert_file.write_text(cert_data.get('certificate', ''))
                    key_file.write_text(cert_data.get('private_key', ''))
                    
                    self._cert_file = cert_file
                    self._key_file = key_file
                    self._enabled = True
                    
                    logger.info(f"HTTPS证书申请成功: {domain}")
                    return True
            
            # 生成自签名证书（开发环境）
            logger.warning(f"生成自签名证书 for {domain} (仅用于开发)")
            self._generate_self_signed_cert(domain)
            
            self._enabled = True
            return True
            
        except Exception as e:
            logger.error(f"启用HTTPS失败: {e}")
            return False
    
    def _generate_self_signed_cert(self, domain: str):
        """生成自签名证书"""
        cert_file = self.cert_dir / f"{domain}.crt"
        key_file = self.cert_dir / f"{domain}.key"
        
        # 使用openssl生成自签名证书
        try:
            subprocess.run([
                "openssl", "req", "-x509", "-newkey", "rsa:4096",
                "-keyout", str(key_file),
                "-out", str(cert_file),
                "-days", "365",
                "-nodes", "-subj",
                f"/C=CN/ST=State/L=City/O=Organization/CN={domain}"
            ], check=True, capture_output=True)
            
            self._cert_file = cert_file
            self._key_file = key_file
            
            logger.info(f"自签名证书生成成功: {cert_file}")
        except subprocess.CalledProcessError as e:
            logger.error(f"生成自签名证书失败: {e}")
            raise
        except FileNotFoundError:
            logger.warning("未找到openssl，无法生成自签名证书")
    
    def get_ssl_context(self) -> Optional[ssl.SSLContext]:
        """获取SSL上下文"""
        if not self._enabled or not self._cert_file or not self._key_file:
            return None
        
        try:
            context = ssl.SSLContext(ssl.PROTOCOL_TLS_SERVER)
            context.load_cert_chain(
                str(self._cert_file),
                str(self._key_file)
            )
            
            # 安全配置
            context.set_ciphers('HIGH:!aNULL:!MD5')
            context.options |= ssl.OP_NO_SSLv2
            context.options |= ssl.OP_NO_SSLv3
            
            return context
        except Exception as e:
            logger.error(f"创建SSL上下文失败: {e}")
            return None
    
    def is_enabled(self) -> bool:
        """检查HTTPS是否启用"""
        return self._enabled
    
    def get_cert_info(self) -> Optional[Dict]:
        """获取证书信息"""
        if not self._cert_file or not self._cert_file.exists():
            return None
        
        try:
            return self.ssl_manager.get_certificate_info(str(self._cert_file))
        except Exception as e:
            logger.error(f"获取证书信息失败: {e}")
            return None
    
    def renew_certificate(self, domain: str, email: str) -> bool:
        """续期证书"""
        try:
            logger.info(f"开始续期证书: {domain}")
            cert_data = self.ssl_manager.renew_certificate(domain, email)
            
            if cert_data:
                cert_file = self.cert_dir / f"{domain}.crt"
                key_file = self.cert_dir / f"{domain}.key"
                
                cert_file.write_text(cert_data.get('certificate', ''))
                key_file.write_text(cert_data.get('private_key', ''))
                
                logger.info(f"证书续期成功: {domain}")
                return True
            
            return False
        except Exception as e:
            logger.error(f"证书续期失败: {e}")
            return False

