"""
防火墙管理器 - 配置和管理防火墙规则
"""
import ipaddress
from typing import List, Dict, Optional
from dataclasses import dataclass
from enum import Enum
from loguru import logger
import platform


class FirewallAction(str, Enum):
    """防火墙动作"""
    ALLOW = "allow"
    DENY = "deny"
    REJECT = "reject"


@dataclass
class FirewallRule:
    """防火墙规则"""
    name: str
    action: FirewallAction
    protocol: str  # tcp, udp, icmp, all
    port: Optional[int] = None
    source_ip: Optional[str] = None
    destination_ip: Optional[str] = None
    enabled: bool = True


class FirewallManager:
    """防火墙管理器"""
    
    def __init__(self):
        self.rules: List[FirewallRule] = []
        self._enabled = False
        self._initialize_default_rules()
    
    def _initialize_default_rules(self):
        """初始化默认规则"""
        # 允许本地回环
        self.add_rule(FirewallRule(
            name="allow_localhost",
            action=FirewallAction.ALLOW,
            protocol="all",
            source_ip="127.0.0.1"
        ))
        
        # 允许内网
        self.add_rule(FirewallRule(
            name="allow_private_network",
            action=FirewallAction.ALLOW,
            protocol="all",
            source_ip="10.0.0.0/8"
        ))
        
        self.add_rule(FirewallRule(
            name="allow_private_network_172",
            action=FirewallAction.ALLOW,
            protocol="all",
            source_ip="172.16.0.0/12"
        ))
        
        self.add_rule(FirewallRule(
            name="allow_private_network_192",
            action=FirewallAction.ALLOW,
            protocol="all",
            source_ip="192.168.0.0/16"
        ))
        
        logger.info("防火墙默认规则初始化完成")
    
    def enable(self):
        """启用防火墙"""
        self._enabled = True
        logger.info("防火墙已启用")
    
    def disable(self):
        """禁用防火墙"""
        self._enabled = False
        logger.info("防火墙已禁用")
    
    def add_rule(self, rule: FirewallRule):
        """添加规则"""
        self.rules.append(rule)
        logger.info(f"添加防火墙规则: {rule.name} ({rule.action.value})")
    
    def remove_rule(self, name: str):
        """移除规则"""
        self.rules = [r for r in self.rules if r.name != name]
        logger.info(f"移除防火墙规则: {name}")
    
    def check_access(self, protocol: str, port: int, source_ip: str, 
                    destination_ip: str = "0.0.0.0") -> bool:
        """检查访问权限"""
        if not self._enabled:
            return True  # 防火墙未启用，允许所有
        
        # 按顺序检查规则
        for rule in self.rules:
            if not rule.enabled:
                continue
            
            # 检查协议
            if rule.protocol != "all" and rule.protocol != protocol:
                continue
            
            # 检查端口
            if rule.port and rule.port != port:
                continue
            
            # 检查源IP
            if rule.source_ip:
                try:
                    if source_ip not in ipaddress.ip_network(rule.source_ip, strict=False):
                        continue
                except ValueError:
                    # IP格式错误，跳过
                    continue
            
            # 检查目标IP
            if rule.destination_ip:
                try:
                    if destination_ip not in ipaddress.ip_network(rule.destination_ip, strict=False):
                        continue
                except ValueError:
                    continue
            
            # 匹配规则，返回动作
            if rule.action == FirewallAction.ALLOW:
                return True
            elif rule.action in [FirewallAction.DENY, FirewallAction.REJECT]:
                logger.warning(f"防火墙阻止访问: {source_ip}:{port} ({protocol}) - 规则: {rule.name}")
                return False
        
        # 默认拒绝（白名单模式）
        logger.warning(f"防火墙默认拒绝: {source_ip}:{port} ({protocol})")
        return False
    
    def apply_system_firewall(self):
        """应用系统防火墙规则（需要root权限）"""
        system = platform.system()
        
        if system == "Linux":
            self._apply_iptables_rules()
        elif system == "Darwin":  # macOS
            self._apply_pfctl_rules()
        else:
            logger.warning(f"不支持的系统防火墙: {system}")
    
    def _apply_iptables_rules(self):
        """应用iptables规则（Linux）"""
        try:
            import subprocess
            
            # 清空现有规则（谨慎使用）
            # subprocess.run(["iptables", "-F"], check=True)
            
            # 添加规则
            for rule in self.rules:
                if not rule.enabled:
                    continue
                
                cmd = ["iptables", "-A", "INPUT"]
                
                if rule.protocol != "all":
                    cmd.extend(["-p", rule.protocol])
                
                if rule.port:
                    cmd.extend(["--dport", str(rule.port)])
                
                if rule.source_ip:
                    cmd.extend(["-s", rule.source_ip])
                
                if rule.action == FirewallAction.ALLOW:
                    cmd.extend(["-j", "ACCEPT"])
                elif rule.action == FirewallAction.DENY:
                    cmd.extend(["-j", "DROP"])
                elif rule.action == FirewallAction.REJECT:
                    cmd.extend(["-j", "REJECT"])
                
                try:
                    subprocess.run(cmd, check=True, capture_output=True)
                    logger.info(f"应用iptables规则: {rule.name}")
                except subprocess.CalledProcessError as e:
                    logger.error(f"应用iptables规则失败: {rule.name} - {e}")
        
        except Exception as e:
            logger.error(f"应用系统防火墙规则失败: {e}")
    
    def _apply_pfctl_rules(self):
        """应用pfctl规则（macOS）"""
        logger.info("macOS防火墙规则需要手动配置pf.conf")
    
    def get_rules(self) -> List[Dict]:
        """获取所有规则"""
        return [
            {
                'name': rule.name,
                'action': rule.action.value,
                'protocol': rule.protocol,
                'port': rule.port,
                'source_ip': rule.source_ip,
                'destination_ip': rule.destination_ip,
                'enabled': rule.enabled,
            }
            for rule in self.rules
        ]
    
    def get_status(self) -> Dict:
        """获取防火墙状态"""
        return {
            'enabled': self._enabled,
            'rule_count': len(self.rules),
            'enabled_rules': len([r for r in self.rules if r.enabled]),
        }

