"""
深度防御 - 多层安全防护
"""
from typing import Dict, List, Optional
from dataclasses import dataclass
from datetime import datetime, timedelta
from enum import Enum
from loguru import logger
from collections import defaultdict
import asyncio


class DefenseLayer(str, Enum):
    """防御层"""
    NETWORK = "network"  # 网络层
    APPLICATION = "application"  # 应用层
    DATA = "data"  # 数据层
    ACCESS = "access"  # 访问层


class ThreatLevel(str, Enum):
    """威胁级别"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


@dataclass
class SecurityEvent:
    """安全事件"""
    event_id: str
    layer: DefenseLayer
    threat_level: ThreatLevel
    event_type: str
    source_ip: str
    description: str
    timestamp: datetime
    blocked: bool = False
    metadata: Dict = None


class DefenseInDepth:
    """深度防御系统"""
    
    def __init__(self):
        self.layers: Dict[DefenseLayer, List[callable]] = {
            DefenseLayer.NETWORK: [],
            DefenseLayer.APPLICATION: [],
            DefenseLayer.DATA: [],
            DefenseLayer.ACCESS: [],
        }
        self.security_events: List[SecurityEvent] = []
        self.threat_intelligence: Dict[str, ThreatLevel] = {}
        self._enabled = True
    
    def add_layer_protection(self, layer: DefenseLayer, protection_func: callable):
        """添加层防护"""
        self.layers[layer].append(protection_func)
        logger.info(f"添加{layer.value}层防护")
    
    async def check_request(self, request_data: Dict) -> tuple[bool, Optional[SecurityEvent]]:
        """检查请求（多层检查）"""
        source_ip = request_data.get('source_ip', 'unknown')
        
        # 逐层检查
        for layer in DefenseLayer:
            for protection_func in self.layers[layer]:
                try:
                    if asyncio.iscoroutinefunction(protection_func):
                        result = await protection_func(request_data)
                    else:
                        result = protection_func(request_data)
                    
                    if not result:
                        # 检测到威胁
                        threat_level = self._assess_threat_level(layer, source_ip)
                        
                        event = SecurityEvent(
                            event_id=self._generate_event_id(),
                            layer=layer,
                            threat_level=threat_level,
                            event_type="blocked",
                            source_ip=source_ip,
                            description=f"{layer.value}层防护阻止了请求",
                            timestamp=datetime.now(),
                            blocked=True,
                            metadata=request_data
                        )
                        
                        self.security_events.append(event)
                        
                        # 只保留最近10000条事件
                        if len(self.security_events) > 10000:
                            self.security_events = self.security_events[-10000:]
                        
                        logger.warning(f"安全事件: {event.description} (来源: {source_ip})")
                        return False, event
                
                except Exception as e:
                    logger.error(f"防护检查失败: {layer.value} - {e}")
        
        # 所有层检查通过
        return True, None
    
    def _assess_threat_level(self, layer: DefenseLayer, source_ip: str) -> ThreatLevel:
        """评估威胁级别"""
        # 检查威胁情报
        if source_ip in self.threat_intelligence:
            return self.threat_intelligence[source_ip]
        
        # 检查历史事件
        recent_events = [
            e for e in self.security_events
            if e.source_ip == source_ip and
            (datetime.now() - e.timestamp).total_seconds() < 3600
        ]
        
        if len(recent_events) >= 10:
            return ThreatLevel.CRITICAL
        elif len(recent_events) >= 5:
            return ThreatLevel.HIGH
        elif len(recent_events) >= 2:
            return ThreatLevel.MEDIUM
        else:
            return ThreatLevel.LOW
    
    def _generate_event_id(self) -> str:
        """生成事件ID"""
        import uuid
        return str(uuid.uuid4())
    
    def add_threat_intelligence(self, source_ip: str, threat_level: ThreatLevel):
        """添加威胁情报"""
        self.threat_intelligence[source_ip] = threat_level
        logger.info(f"添加威胁情报: {source_ip} -> {threat_level.value}")
    
    def get_security_report(self, hours: int = 24) -> Dict:
        """获取安全报告"""
        cutoff_time = datetime.now() - timedelta(hours=hours)
        
        recent_events = [
            e for e in self.security_events
            if e.timestamp >= cutoff_time
        ]
        
        report = {
            'total_events': len(recent_events),
            'blocked_events': len([e for e in recent_events if e.blocked]),
            'by_layer': defaultdict(int),
            'by_threat_level': defaultdict(int),
            'top_threats': [],
        }
        
        for event in recent_events:
            report['by_layer'][event.layer.value] += 1
            report['by_threat_level'][event.threat_level.value] += 1
        
        # Top威胁源
        threat_sources = defaultdict(int)
        for event in recent_events:
            if event.blocked:
                threat_sources[event.source_ip] += 1
        
        report['top_threats'] = [
            {'ip': ip, 'count': count}
            for ip, count in sorted(threat_sources.items(), key=lambda x: x[1], reverse=True)[:10]
        ]
        
        return report


# 预定义的防护函数
async def network_layer_protection(request_data: Dict) -> bool:
    """网络层防护"""
    from .ip_filter import IPFilter
    
    source_ip = request_data.get('source_ip')
    if not source_ip:
        return True
    
    ip_filter = IPFilter()
    return ip_filter.is_allowed(source_ip)


async def application_layer_protection(request_data: Dict) -> bool:
    """应用层防护"""
    from .rate_limiter import APIRateLimiter
    
    endpoint = request_data.get('endpoint', '')
    source_ip = request_data.get('source_ip', '')
    
    rate_limiter = APIRateLimiter()
    key = f"ip:{source_ip}"
    return rate_limiter.check_limit(key)


async def data_layer_protection(request_data: Dict) -> bool:
    """数据层防护"""
    # 检查数据大小
    data_size = len(str(request_data.get('data', '')))
    if data_size > 10 * 1024 * 1024:  # 10MB限制
        return False
    
    # 检查恶意内容（简化实现）
    data_str = str(request_data.get('data', '')).lower()
    malicious_patterns = ['<script', 'javascript:', 'eval(']
    
    for pattern in malicious_patterns:
        if pattern in data_str:
            return False
    
    return True


async def access_layer_protection(request_data: Dict) -> bool:
    """访问层防护"""
    # 检查认证和授权
    token = request_data.get('token')
    if not token:
        return False
    
    # 这里应该验证token
    # 简化实现
    return True

