"""
配额管理器
"""
from typing import Dict, Optional
from datetime import datetime, timedelta
from dataclasses import dataclass
from loguru import logger
from enum import Enum


class QuotaType(str, Enum):
    """配额类型"""
    TRAFFIC = "traffic"  # 流量配额
    CONNECTIONS = "connections"  # 连接数配额
    TIME = "time"  # 时间配额


class QuotaPeriod(str, Enum):
    """配额周期"""
    DAILY = "daily"
    WEEKLY = "weekly"
    MONTHLY = "monthly"
    UNLIMITED = "unlimited"


@dataclass
class Quota:
    """配额配置"""
    client_id: str
    quota_type: QuotaType
    limit: int  # 配额限制（字节或连接数）
    period: QuotaPeriod
    used: int = 0  # 已使用量
    reset_at: Optional[datetime] = None  # 重置时间
    
    def is_exceeded(self) -> bool:
        """检查是否超出配额"""
        return self.used >= self.limit
    
    def get_remaining(self) -> int:
        """获取剩余配额"""
        return max(0, self.limit - self.used)
    
    def get_usage_percent(self) -> float:
        """获取使用百分比"""
        if self.limit == 0:
            return 0.0
        return min(100.0, (self.used / self.limit) * 100)


class QuotaManager:
    """配额管理器"""
    
    def __init__(self):
        self._quotas: Dict[str, Dict[QuotaType, Quota]] = {}  # client_id -> {type: quota}
        self._warnings_sent: Dict[str, set] = {}  # client_id -> {warning_types}
    
    def set_quota(self, client_id: str, quota_type: QuotaType, limit: int, period: QuotaPeriod = QuotaPeriod.MONTHLY):
        """设置配额"""
        if client_id not in self._quotas:
            self._quotas[client_id] = {}
        
        reset_at = self._calculate_reset_time(period)
        
        quota = Quota(
            client_id=client_id,
            quota_type=quota_type,
            limit=limit,
            period=period,
            reset_at=reset_at,
        )
        
        self._quotas[client_id][quota_type] = quota
        logger.info(f"设置配额: {client_id} {quota_type.value} = {limit} ({period.value})")
    
    def get_quota(self, client_id: str, quota_type: QuotaType) -> Optional[Quota]:
        """获取配额"""
        if client_id not in self._quotas:
            return None
        return self._quotas[client_id].get(quota_type)
    
    def check_quota(self, client_id: str, quota_type: QuotaType, amount: int = 1) -> tuple[bool, Optional[str]]:
        """检查配额
        
        Returns:
            (是否允许, 错误消息)
        """
        quota = self.get_quota(client_id, quota_type)
        
        if not quota:
            return True, None  # 无配额限制
        
        # 检查是否需要重置
        if quota.reset_at and datetime.now() >= quota.reset_at:
            self._reset_quota(quota)
        
        # 检查是否超出配额
        if quota.is_exceeded():
            return False, f"配额已用完 ({quota_type.value})"
        
        # 检查是否接近配额限制（80%警告）
        if quota.get_usage_percent() >= 80 and quota_type not in self._warnings_sent.get(client_id, set()):
            if client_id not in self._warnings_sent:
                self._warnings_sent[client_id] = set()
            self._warnings_sent[client_id].add(quota_type)
            logger.warning(f"客户端 {client_id} {quota_type.value} 配额使用超过80%")
        
        # 检查是否会超出配额
        if quota.used + amount > quota.limit:
            return False, f"配额不足 ({quota_type.value})"
        
        return True, None
    
    def consume_quota(self, client_id: str, quota_type: QuotaType, amount: int):
        """消耗配额"""
        quota = self.get_quota(client_id, quota_type)
        if quota:
            quota.used += amount
            logger.debug(f"消耗配额: {client_id} {quota_type.value} +{amount} (已用: {quota.used}/{quota.limit})")
    
    def _reset_quota(self, quota: Quota):
        """重置配额"""
        quota.used = 0
        quota.reset_at = self._calculate_reset_time(quota.period)
        logger.info(f"重置配额: {quota.client_id} {quota.quota_type.value}")
    
    def _calculate_reset_time(self, period: QuotaPeriod) -> Optional[datetime]:
        """计算重置时间"""
        now = datetime.now()
        
        if period == QuotaPeriod.DAILY:
            return (now + timedelta(days=1)).replace(hour=0, minute=0, second=0, microsecond=0)
        elif period == QuotaPeriod.WEEKLY:
            days_until_monday = (7 - now.weekday()) % 7
            if days_until_monday == 0:
                days_until_monday = 7
            return (now + timedelta(days=days_until_monday)).replace(hour=0, minute=0, second=0, microsecond=0)
        elif period == QuotaPeriod.MONTHLY:
            # 下个月的第一天
            if now.month == 12:
                return datetime(now.year + 1, 1, 1)
            else:
                return datetime(now.year, now.month + 1, 1)
        else:
            return None  # UNLIMITED
    
    def get_client_quotas(self, client_id: str) -> Dict[QuotaType, Quota]:
        """获取客户端的所有配额"""
        return self._quotas.get(client_id, {})
    
    def remove_client_quotas(self, client_id: str):
        """移除客户端的所有配额"""
        if client_id in self._quotas:
            del self._quotas[client_id]
        if client_id in self._warnings_sent:
            del self._warnings_sent[client_id]

