"""
数据库查询优化器
"""
from typing import List, Dict, Optional
from sqlalchemy.orm import Query
from sqlalchemy import text
from loguru import logger
import time


class QueryOptimizer:
    """查询优化器"""
    
    def __init__(self):
        self.slow_query_threshold = 1.0  # 1秒
        self.query_stats: Dict[str, List[float]] = {}
    
    def optimize_query(self, query: Query, use_index: bool = True, 
                      limit: Optional[int] = None) -> Query:
        """优化查询"""
        # 添加索引提示
        if use_index:
            # 这里可以根据实际情况添加索引提示
            pass
        
        # 添加限制
        if limit:
            query = query.limit(limit)
        
        return query
    
    def measure_query_time(self, query_func, query_name: str = "unknown"):
        """测量查询时间"""
        start_time = time.time()
        result = query_func()
        elapsed = time.time() - start_time
        
        # 记录慢查询
        if elapsed > self.slow_query_threshold:
            logger.warning(f"慢查询检测: {query_name} 耗时 {elapsed:.3f}秒")
        
        # 记录统计
        if query_name not in self.query_stats:
            self.query_stats[query_name] = []
        self.query_stats[query_name].append(elapsed)
        
        return result, elapsed
    
    def get_query_statistics(self) -> Dict:
        """获取查询统计"""
        stats = {}
        
        for query_name, times in self.query_stats.items():
            if times:
                stats[query_name] = {
                    'count': len(times),
                    'avg': sum(times) / len(times),
                    'min': min(times),
                    'max': max(times),
                    'total': sum(times),
                }
        
        return stats
    
    def analyze_query_plan(self, query: Query, session):
        """分析查询计划"""
        try:
            # 获取SQL
            sql = str(query.statement.compile(compile_kwargs={"literal_binds": True}))
            
            # 使用EXPLAIN分析
            explain_query = text(f"EXPLAIN ANALYZE {sql}")
            result = session.execute(explain_query)
            
            plan = []
            for row in result:
                plan.append(str(row))
            
            return {
                'sql': sql,
                'plan': plan
            }
        except Exception as e:
            logger.error(f"分析查询计划失败: {e}")
            return None
    
    def suggest_indexes(self, table_name: str, columns: List[str]) -> List[str]:
        """建议索引"""
        suggestions = []
        
        # 为常用查询列建议索引
        for column in columns:
            suggestions.append(f"CREATE INDEX IF NOT EXISTS idx_{table_name}_{column} ON {table_name}({column});")
        
        return suggestions
    
    def optimize_bulk_operations(self, operations: List[callable], batch_size: int = 1000):
        """优化批量操作"""
        # 分批处理
        for i in range(0, len(operations), batch_size):
            batch = operations[i:i + batch_size]
            yield batch

