"""
连接池优化 - 数据库和HTTP连接池管理
"""
import asyncio
from typing import Optional, Dict
from contextlib import asynccontextmanager
from loguru import logger
from sqlalchemy.pool import QueuePool
from sqlalchemy import create_engine, event
import aiohttp


class DatabaseConnectionPool:
    """数据库连接池优化"""
    
    def __init__(self, database_url: str, pool_size: int = 20, max_overflow: int = 40):
        self.database_url = database_url
        self.pool_size = pool_size
        self.max_overflow = max_overflow
        self.engine = None
        self._initialize_pool()
    
    def _initialize_pool(self):
        """初始化连接池"""
        self.engine = create_engine(
            self.database_url,
            poolclass=QueuePool,
            pool_size=self.pool_size,
            max_overflow=self.max_overflow,
            pool_pre_ping=True,  # 连接前ping检查
            pool_recycle=3600,  # 1小时回收连接
            echo=False,
        )
        
        # 监听连接事件
        @event.listens_for(self.engine, "connect")
        def set_sqlite_pragma(dbapi_conn, connection_record):
            """设置数据库连接参数"""
            # PostgreSQL优化参数
            if 'postgresql' in self.database_url:
                with dbapi_conn.cursor() as cursor:
                    cursor.execute("SET statement_timeout = 30000")  # 30秒超时
                    cursor.execute("SET idle_in_transaction_session_timeout = 60000")  # 60秒
        
        logger.info(f"数据库连接池初始化完成 (pool_size={self.pool_size}, max_overflow={self.max_overflow})")
    
    def get_engine(self):
        """获取引擎"""
        return self.engine
    
    def get_pool_status(self) -> Dict:
        """获取连接池状态"""
        pool = self.engine.pool
        return {
            'size': pool.size(),
            'checked_in': pool.checkedin(),
            'checked_out': pool.checkedout(),
            'overflow': pool.overflow(),
            'invalid': pool.invalid(),
        }


class HTTPConnectionPool:
    """HTTP连接池优化"""
    
    def __init__(self, max_connections: int = 100, max_connections_per_host: int = 30):
        self.max_connections = max_connections
        self.max_connections_per_host = max_connections_per_host
        self.connector: Optional[aiohttp.TCPConnector] = None
        self.session: Optional[aiohttp.ClientSession] = None
        self._initialize_pool()
    
    def _initialize_pool(self):
        """初始化HTTP连接池"""
        self.connector = aiohttp.TCPConnector(
            limit=self.max_connections,
            limit_per_host=self.max_connections_per_host,
            ttl_dns_cache=300,  # DNS缓存5分钟
            use_dns_cache=True,
            keepalive_timeout=30,
            enable_cleanup_closed=True,
        )
        
        timeout = aiohttp.ClientTimeout(
            total=30,
            connect=10,
            sock_read=10
        )
        
        self.session = aiohttp.ClientSession(
            connector=self.connector,
            timeout=timeout,
            connector_owner=False
        )
        
        logger.info(f"HTTP连接池初始化完成 (max_connections={self.max_connections})")
    
    async def get_session(self) -> aiohttp.ClientSession:
        """获取会话"""
        if self.session is None or self.session.closed:
            self._initialize_pool()
        return self.session
    
    async def close(self):
        """关闭连接池"""
        if self.session:
            await self.session.close()
        if self.connector:
            await self.connector.close()
    
    def get_pool_status(self) -> Dict:
        """获取连接池状态"""
        if not self.connector:
            return {}
        
        return {
            'limit': self.connector.limit,
            'limit_per_host': self.connector.limit_per_host,
            'connections': len(self.connector._conns) if hasattr(self.connector, '_conns') else 0,
        }


class ConnectionPoolManager:
    """连接池管理器"""
    
    def __init__(self):
        self.db_pool: Optional[DatabaseConnectionPool] = None
        self.http_pool: Optional[HTTPConnectionPool] = None
    
    def init_database_pool(self, database_url: str, pool_size: int = 20, max_overflow: int = 40):
        """初始化数据库连接池"""
        self.db_pool = DatabaseConnectionPool(database_url, pool_size, max_overflow)
    
    def init_http_pool(self, max_connections: int = 100, max_connections_per_host: int = 30):
        """初始化HTTP连接池"""
        self.http_pool = HTTPConnectionPool(max_connections, max_connections_per_host)
    
    async def close_all(self):
        """关闭所有连接池"""
        if self.http_pool:
            await self.http_pool.close()
    
    def get_status(self) -> Dict:
        """获取所有连接池状态"""
        status = {}
        
        if self.db_pool:
            status['database'] = self.db_pool.get_pool_status()
        
        if self.http_pool:
            status['http'] = self.http_pool.get_pool_status()
        
        return status

