"""
健康检查器 - 自动化健康检查和恢复
"""
import asyncio
import aiohttp
from typing import Dict, List, Optional, Callable
from dataclasses import dataclass
from datetime import datetime, timedelta
from enum import Enum
from loguru import logger


class HealthStatus(str, Enum):
    """健康状态"""
    HEALTHY = "healthy"
    UNHEALTHY = "unhealthy"
    DEGRADED = "degraded"
    UNKNOWN = "unknown"


@dataclass
class HealthCheck:
    """健康检查"""
    name: str
    check_func: Callable
    interval: int = 30  # 检查间隔（秒）
    timeout: int = 5  # 超时时间（秒）
    retry_count: int = 3  # 重试次数
    enabled: bool = True
    last_check: Optional[datetime] = None
    last_status: HealthStatus = HealthStatus.UNKNOWN
    consecutive_failures: int = 0


class HealthChecker:
    """健康检查器"""
    
    def __init__(self):
        self.checks: Dict[str, HealthCheck] = {}
        self._running = False
        self._tasks: List[asyncio.Task] = []
        self.recovery_callbacks: List[Callable] = []
    
    def add_check(self, check: HealthCheck):
        """添加健康检查"""
        self.checks[check.name] = check
        logger.info(f"添加健康检查: {check.name}")
    
    async def check_health(self, check_name: str) -> HealthStatus:
        """执行健康检查"""
        check = self.checks.get(check_name)
        if not check or not check.enabled:
            return HealthStatus.UNKNOWN
        
        try:
            # 执行检查函数
            if asyncio.iscoroutinefunction(check.check_func):
                result = await asyncio.wait_for(
                    check.check_func(),
                    timeout=check.timeout
                )
            else:
                result = check.check_func()
            
            check.last_check = datetime.now()
            
            # 判断结果
            if result:
                check.last_status = HealthStatus.HEALTHY
                check.consecutive_failures = 0
            else:
                check.consecutive_failures += 1
                if check.consecutive_failures >= check.retry_count:
                    check.last_status = HealthStatus.UNHEALTHY
                else:
                    check.last_status = HealthStatus.DEGRADED
            
            return check.last_status
            
        except asyncio.TimeoutError:
            check.consecutive_failures += 1
            check.last_status = HealthStatus.UNHEALTHY
            logger.warning(f"健康检查超时: {check.name}")
            return HealthStatus.UNHEALTHY
        
        except Exception as e:
            check.consecutive_failures += 1
            check.last_status = HealthStatus.UNHEALTHY
            logger.error(f"健康检查失败: {check.name} - {e}")
            return HealthStatus.UNHEALTHY
    
    async def check_all(self) -> Dict[str, HealthStatus]:
        """检查所有健康状态"""
        results = {}
        
        for check_name in self.checks.keys():
            status = await self.check_health(check_name)
            results[check_name] = status
            
            # 如果状态不健康，尝试恢复
            if status == HealthStatus.UNHEALTHY:
                await self._attempt_recovery(check_name)
        
        return results
    
    async def _attempt_recovery(self, check_name: str):
        """尝试恢复"""
        check = self.checks.get(check_name)
        if not check:
            return
        
        logger.info(f"尝试恢复: {check_name}")
        
        # 调用恢复回调
        for callback in self.recovery_callbacks:
            try:
                if asyncio.iscoroutinefunction(callback):
                    await callback(check_name, check)
                else:
                    callback(check_name, check)
            except Exception as e:
                logger.error(f"恢复回调执行失败: {e}")
    
    async def monitor_loop(self, check_name: str):
        """监控循环"""
        check = self.checks.get(check_name)
        if not check:
            return
        
        while self._running:
            try:
                await self.check_health(check_name)
                await asyncio.sleep(check.interval)
            except Exception as e:
                logger.error(f"监控循环错误: {check_name} - {e}")
                await asyncio.sleep(check.interval)
    
    async def start_monitoring(self):
        """启动监控"""
        self._running = True
        
        for check_name in self.checks.keys():
            task = asyncio.create_task(self.monitor_loop(check_name))
            self._tasks.append(task)
        
        logger.info(f"启动了 {len(self._tasks)} 个健康检查监控")
    
    async def stop_monitoring(self):
        """停止监控"""
        self._running = False
        
        if self._tasks:
            for task in self._tasks:
                task.cancel()
            await asyncio.gather(*self._tasks, return_exceptions=True)
            self._tasks.clear()
    
    def add_recovery_callback(self, callback: Callable):
        """添加恢复回调"""
        self.recovery_callbacks.append(callback)
    
    def get_overall_health(self) -> HealthStatus:
        """获取整体健康状态"""
        if not self.checks:
            return HealthStatus.UNKNOWN
        
        statuses = [check.last_status for check in self.checks.values() if check.enabled]
        
        if not statuses:
            return HealthStatus.UNKNOWN
        
        if HealthStatus.UNHEALTHY in statuses:
            return HealthStatus.UNHEALTHY
        elif HealthStatus.DEGRADED in statuses:
            return HealthStatus.DEGRADED
        elif all(s == HealthStatus.HEALTHY for s in statuses):
            return HealthStatus.HEALTHY
        else:
            return HealthStatus.UNKNOWN
    
    def get_health_report(self) -> Dict:
        """获取健康报告"""
        return {
            'overall': self.get_overall_health().value,
            'checks': {
                name: {
                    'status': check.last_status.value,
                    'last_check': check.last_check.isoformat() if check.last_check else None,
                    'consecutive_failures': check.consecutive_failures,
                    'enabled': check.enabled,
                }
                for name, check in self.checks.items()
            }
        }


# 预定义的健康检查函数
async def check_database_health():
    """检查数据库健康"""
    try:
        from ..db.database import get_db
        async with get_db() as db:
            await db.execute("SELECT 1")
        return True
    except Exception:
        return False


async def check_redis_health():
    """检查Redis健康"""
    try:
        import os
        import redis.asyncio as aioredis
        redis_url = os.getenv("REDIS_URL", "redis://localhost:6379/0")
        redis_client = await aioredis.from_url(redis_url)
        await redis_client.ping()
        await redis_client.close()
        return True
    except Exception:
        return False


async def check_api_health():
    """检查API健康"""
    try:
        import os
        import aiohttp
        base_url = os.getenv("BASE_URL", "http://localhost:8080")
        async with aiohttp.ClientSession() as session:
            async with session.get(f"{base_url}/api/health", timeout=aiohttp.ClientTimeout(total=5)) as resp:
                return resp.status == 200
    except Exception:
        return False

