"""
多租户管理器
"""
from typing import Dict, List, Optional
from dataclasses import dataclass
from datetime import datetime
from loguru import logger
import uuid


@dataclass
class Tenant:
    """租户"""
    tenant_id: str
    name: str
    domain: Optional[str] = None
    status: str = "active"  # active, suspended, deleted
    created_at: datetime = None
    metadata: Dict = None
    
    def __post_init__(self):
        if self.created_at is None:
            self.created_at = datetime.now()
        if self.metadata is None:
            self.metadata = {}


class TenantManager:
    """多租户管理器"""
    
    def __init__(self):
        self.tenants: Dict[str, Tenant] = {}
        self.tenant_resources: Dict[str, Dict] = {}  # tenant_id -> resources
    
    def create_tenant(self, name: str, domain: Optional[str] = None, 
                     metadata: Dict = None) -> Tenant:
        """创建租户"""
        tenant_id = str(uuid.uuid4())
        
        tenant = Tenant(
            tenant_id=tenant_id,
            name=name,
            domain=domain,
            metadata=metadata or {}
        )
        
        self.tenants[tenant_id] = tenant
        self.tenant_resources[tenant_id] = {
            'clients': [],
            'tunnels': [],
            'domains': [],
        }
        
        logger.info(f"创建租户: {tenant_id} ({name})")
        return tenant
    
    def get_tenant(self, tenant_id: str) -> Optional[Tenant]:
        """获取租户"""
        return self.tenants.get(tenant_id)
    
    def get_tenant_by_domain(self, domain: str) -> Optional[Tenant]:
        """根据域名获取租户"""
        for tenant in self.tenants.values():
            if tenant.domain == domain:
                return tenant
        return None
    
    def update_tenant(self, tenant_id: str, **kwargs) -> bool:
        """更新租户"""
        tenant = self.tenants.get(tenant_id)
        if not tenant:
            return False
        
        for key, value in kwargs.items():
            if hasattr(tenant, key):
                setattr(tenant, key, value)
        
        logger.info(f"更新租户: {tenant_id}")
        return True
    
    def delete_tenant(self, tenant_id: str) -> bool:
        """删除租户"""
        if tenant_id not in self.tenants:
            return False
        
        tenant = self.tenants[tenant_id]
        tenant.status = "deleted"
        
        # 清理资源
        if tenant_id in self.tenant_resources:
            del self.tenant_resources[tenant_id]
        
        logger.info(f"删除租户: {tenant_id}")
        return True
    
    def add_resource(self, tenant_id: str, resource_type: str, resource_id: str):
        """添加资源到租户"""
        if tenant_id not in self.tenant_resources:
            self.tenant_resources[tenant_id] = {
                'clients': [],
                'tunnels': [],
                'domains': [],
            }
        
        if resource_type in self.tenant_resources[tenant_id]:
            if resource_id not in self.tenant_resources[tenant_id][resource_type]:
                self.tenant_resources[tenant_id][resource_type].append(resource_id)
    
    def get_tenant_resources(self, tenant_id: str) -> Dict:
        """获取租户资源"""
        return self.tenant_resources.get(tenant_id, {
            'clients': [],
            'tunnels': [],
            'domains': [],
        })
    
    def list_tenants(self, status: Optional[str] = None) -> List[Tenant]:
        """列出租户"""
        tenants = list(self.tenants.values())
        
        if status:
            tenants = [t for t in tenants if t.status == status]
        
        return tenants
    
    def get_tenant_stats(self, tenant_id: str) -> Dict:
        """获取租户统计"""
        resources = self.get_tenant_resources(tenant_id)
        
        return {
            'tenant_id': tenant_id,
            'clients': len(resources.get('clients', [])),
            'tunnels': len(resources.get('tunnels', [])),
            'domains': len(resources.get('domains', [])),
        }


class TenantContext:
    """租户上下文"""
    
    def __init__(self, tenant_id: str):
        self.tenant_id = tenant_id
    
    def __enter__(self):
        # 设置当前租户上下文
        import contextvars
        tenant_context = contextvars.ContextVar('tenant_id')
        tenant_context.set(self.tenant_id)
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        pass


def get_current_tenant() -> Optional[str]:
    """获取当前租户ID"""
    import contextvars
    try:
        tenant_context = contextvars.ContextVar('tenant_id')
        return tenant_context.get()
    except LookupError:
        return None

