"""
智能告警 - 更智能的告警系统
"""
from typing import Dict, List, Optional, Callable
from dataclasses import dataclass
from datetime import datetime, timedelta
from enum import Enum
from loguru import logger
from collections import deque
import asyncio


class AlertSeverity(str, Enum):
    """告警严重程度"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


@dataclass
class AlertCondition:
    """告警条件"""
    metric_name: str
    operator: str  # >, <, >=, <=, ==, !=
    threshold: float
    duration: int  # 持续时间（秒）
    severity: AlertSeverity = AlertSeverity.MEDIUM


class SmartAlertEngine:
    """智能告警引擎"""
    
    def __init__(self):
        self.conditions: List[AlertCondition] = []
        self.metric_history: Dict[str, deque] = {}
        self.alert_history: List[Dict] = []
        self.alert_callbacks: List[Callable] = []
        self._enabled = True
    
    def add_condition(self, condition: AlertCondition):
        """添加告警条件"""
        self.conditions.append(condition)
        
        # 初始化指标历史
        if condition.metric_name not in self.metric_history:
            self.metric_history[condition.metric_name] = deque(maxlen=1000)
        
        logger.info(f"添加告警条件: {condition.metric_name} {condition.operator} {condition.threshold}")
    
    def record_metric(self, metric_name: str, value: float, timestamp: datetime = None):
        """记录指标值"""
        if timestamp is None:
            timestamp = datetime.now()
        
        if metric_name not in self.metric_history:
            self.metric_history[metric_name] = deque(maxlen=1000)
        
        self.metric_history[metric_name].append({
            'value': value,
            'timestamp': timestamp
        })
        
        # 检查告警条件
        self._check_conditions(metric_name)
    
    def _check_conditions(self, metric_name: str):
        """检查告警条件"""
        if not self._enabled:
            return
        
        for condition in self.conditions:
            if condition.metric_name != metric_name:
                continue
            
            history = self.metric_history.get(metric_name, deque())
            if len(history) < 2:
                continue
            
            # 检查持续时间内的值
            now = datetime.now()
            threshold_time = now - timedelta(seconds=condition.duration)
            
            values_in_duration = [
                item['value']
                for item in history
                if item['timestamp'] >= threshold_time
            ]
            
            if not values_in_duration:
                continue
            
            # 检查是否满足条件
            should_alert = False
            
            if condition.operator == ">":
                should_alert = all(v > condition.threshold for v in values_in_duration)
            elif condition.operator == ">=":
                should_alert = all(v >= condition.threshold for v in values_in_duration)
            elif condition.operator == "<":
                should_alert = all(v < condition.threshold for v in values_in_duration)
            elif condition.operator == "<=":
                should_alert = all(v <= condition.threshold for v in values_in_duration)
            elif condition.operator == "==":
                should_alert = all(v == condition.threshold for v in values_in_duration)
            elif condition.operator == "!=":
                should_alert = all(v != condition.threshold for v in values_in_duration)
            
            if should_alert:
                self._trigger_alert(condition, values_in_duration)
    
    def _trigger_alert(self, condition: AlertCondition, values: List[float]):
        """触发告警"""
        # 检查是否已经触发过（避免重复告警）
        recent_alerts = [
            alert for alert in self.alert_history
            if alert['condition'] == condition.metric_name and
            (datetime.now() - alert['timestamp']).total_seconds() < 300  # 5分钟内不重复
        ]
        
        if recent_alerts:
            return
        
        alert = {
            'condition': condition.metric_name,
            'severity': condition.severity.value,
            'threshold': condition.threshold,
            'current_value': values[-1] if values else 0,
            'timestamp': datetime.now(),
            'message': f"{condition.metric_name} {condition.operator} {condition.threshold} (当前值: {values[-1] if values else 0})"
        }
        
        self.alert_history.append(alert)
        
        # 只保留最近1000条告警
        if len(self.alert_history) > 1000:
            self.alert_history = self.alert_history[-1000:]
        
        # 调用回调
        for callback in self.alert_callbacks:
            try:
                if asyncio.iscoroutinefunction(callback):
                    asyncio.create_task(callback(alert))
                else:
                    callback(alert)
            except Exception as e:
                logger.error(f"告警回调执行失败: {e}")
        
        logger.warning(f"触发告警: {alert['message']}")
    
    def add_alert_callback(self, callback: Callable):
        """添加告警回调"""
        self.alert_callbacks.append(callback)
    
    def get_recent_alerts(self, hours: int = 24, severity: Optional[AlertSeverity] = None) -> List[Dict]:
        """获取最近的告警"""
        cutoff_time = datetime.now() - timedelta(hours=hours)
        
        alerts = [
            alert for alert in self.alert_history
            if alert['timestamp'] >= cutoff_time
        ]
        
        if severity:
            alerts = [a for a in alerts if a['severity'] == severity.value]
        
        return sorted(alerts, key=lambda x: x['timestamp'], reverse=True)
    
    def get_alert_statistics(self, hours: int = 24) -> Dict:
        """获取告警统计"""
        alerts = self.get_recent_alerts(hours)
        
        stats = {
            'total': len(alerts),
            'by_severity': {},
            'by_condition': {},
        }
        
        for alert in alerts:
            severity = alert['severity']
            condition = alert['condition']
            
            stats['by_severity'][severity] = stats['by_severity'].get(severity, 0) + 1
            stats['by_condition'][condition] = stats['by_condition'].get(condition, 0) + 1
        
        return stats

