"""
系统监控
"""
import asyncio
import psutil
import time
from typing import Dict, List, Callable, Optional
from datetime import datetime
from loguru import logger

from ..gateway.connection_manager import ConnectionManager
from ..tunnel.tunnel_manager import TunnelManager


class SystemMonitor:
    """系统监控"""
    
    def __init__(
        self,
        connection_manager: ConnectionManager,
        tunnel_manager: TunnelManager,
    ):
        self.connection_manager = connection_manager
        self.tunnel_manager = tunnel_manager
        self._alert_callbacks: List[Callable] = []
        self._monitoring = False
    
    def add_alert_callback(self, callback: Callable):
        """添加告警回调"""
        self._alert_callbacks.append(callback)
    
    async def check_system_health(self) -> dict:
        """检查系统健康状态"""
        try:
            # CPU使用率
            cpu_percent = psutil.cpu_percent(interval=1)
            
            # 内存使用
            memory = psutil.virtual_memory()
            
            # 磁盘使用
            disk = psutil.disk_usage('/')
            
            # 连接数
            connection_count = self.connection_manager.get_connection_count()
            
            # 隧道数
            tunnel_count = len(self.tunnel_manager._tunnels)
            active_tunnel_count = len([
                t for t in self.tunnel_manager._tunnels.values()
                if t.status.value == 'active'
            ])
            
            health = {
                'timestamp': datetime.now().isoformat(),
                'cpu': {
                    'percent': cpu_percent,
                    'status': 'warning' if cpu_percent > 80 else 'ok',
                },
                'memory': {
                    'total': memory.total,
                    'used': memory.used,
                    'percent': memory.percent,
                    'status': 'warning' if memory.percent > 80 else 'ok',
                },
                'disk': {
                    'total': disk.total,
                    'used': disk.used,
                    'percent': disk.percent,
                    'status': 'warning' if disk.percent > 80 else 'ok',
                },
                'connections': {
                    'count': connection_count,
                    'max': 10000,
                    'status': 'warning' if connection_count > 8000 else 'ok',
                },
                'tunnels': {
                    'total': tunnel_count,
                    'active': active_tunnel_count,
                    'status': 'ok',
                },
                'overall_status': 'ok',
            }
            
            # 检查是否有警告
            if any([
                health['cpu']['status'] == 'warning',
                health['memory']['status'] == 'warning',
                health['disk']['status'] == 'warning',
                health['connections']['status'] == 'warning',
            ]):
                health['overall_status'] = 'warning'
            
            return health
            
        except Exception as e:
            logger.error(f"检查系统健康状态失败: {e}")
            return {
                'timestamp': datetime.now().isoformat(),
                'status': 'error',
                'error': str(e),
            }
    
    async def start_monitoring(self, interval: int = 60):
        """启动监控"""
        self._monitoring = True
        
        while self._monitoring:
            try:
                health = await self.check_system_health()
                
                # 检查告警条件
                if health.get('overall_status') == 'warning':
                    await self._trigger_alert('system_warning', health)
                
                await asyncio.sleep(interval)
                
            except Exception as e:
                logger.error(f"监控过程出错: {e}")
                await asyncio.sleep(interval)
    
    def stop_monitoring(self):
        """停止监控"""
        self._monitoring = False
    
    async def _trigger_alert(self, alert_type: str, data: dict):
        """触发告警"""
        for callback in self._alert_callbacks:
            try:
                await callback(alert_type, data)
            except Exception as e:
                logger.error(f"告警回调失败: {e}")

