"""
日志分析器
"""
import re
from typing import List, Dict, Optional
from datetime import datetime, timedelta
from pathlib import Path
from loguru import logger
from collections import defaultdict


class LogAnalyzer:
    """日志分析器"""
    
    def __init__(self, log_dir: str = "logs"):
        self.log_dir = Path(log_dir)
        self.log_dir.mkdir(parents=True, exist_ok=True)
    
    def analyze_logs(self, log_file: str, start_time: Optional[datetime] = None, 
                    end_time: Optional[datetime] = None) -> Dict:
        """分析日志文件"""
        log_path = self.log_dir / log_file
        
        if not log_path.exists():
            logger.warning(f"日志文件不存在: {log_path}")
            return {}
        
        stats = {
            'total_lines': 0,
            'errors': 0,
            'warnings': 0,
            'info': 0,
            'debug': 0,
            'error_types': defaultdict(int),
            'client_activities': defaultdict(int),
            'tunnel_activities': defaultdict(int),
        }
        
        try:
            with open(log_path, 'r', encoding='utf-8') as f:
                for line in f:
                    stats['total_lines'] += 1
                    
                    # 解析日志级别
                    if 'ERROR' in line:
                        stats['errors'] += 1
                        self._extract_error_type(line, stats['error_types'])
                    elif 'WARNING' in line or 'WARN' in line:
                        stats['warnings'] += 1
                    elif 'INFO' in line:
                        stats['info'] += 1
                    elif 'DEBUG' in line:
                        stats['debug'] += 1
                    
                    # 提取客户端活动
                    client_match = re.search(r'client[_\s]+([a-f0-9-]+)', line, re.IGNORECASE)
                    if client_match:
                        stats['client_activities'][client_match.group(1)] += 1
                    
                    # 提取隧道活动
                    tunnel_match = re.search(r'tunnel[_\s]+([a-f0-9-]+)', line, re.IGNORECASE)
                    if tunnel_match:
                        stats['tunnel_activities'][tunnel_match.group(1)] += 1
        
        except Exception as e:
            logger.error(f"分析日志失败: {e}")
        
        return stats
    
    def _extract_error_type(self, line: str, error_types: Dict[str, int]):
        """提取错误类型"""
        # 简单的错误类型提取
        if 'Connection' in line or '连接' in line:
            error_types['Connection'] += 1
        elif 'Timeout' in line or '超时' in line:
            error_types['Timeout'] += 1
        elif 'Authentication' in line or '认证' in line:
            error_types['Authentication'] += 1
        elif 'Permission' in line or '权限' in line:
            error_types['Permission'] += 1
        else:
            error_types['Other'] += 1
    
    def search_logs(self, log_file: str, keyword: str, limit: int = 100) -> List[Dict]:
        """搜索日志"""
        log_path = self.log_dir / log_file
        
        if not log_path.exists():
            return []
        
        results = []
        
        try:
            with open(log_path, 'r', encoding='utf-8') as f:
                for line_num, line in enumerate(f, 1):
                    if keyword.lower() in line.lower():
                        results.append({
                            'line': line_num,
                            'content': line.strip(),
                            'timestamp': self._extract_timestamp(line),
                        })
                        
                        if len(results) >= limit:
                            break
        
        except Exception as e:
            logger.error(f"搜索日志失败: {e}")
        
        return results
    
    def _extract_timestamp(self, line: str) -> Optional[str]:
        """提取时间戳"""
        # 尝试提取ISO格式时间戳
        timestamp_match = re.search(r'(\d{4}-\d{2}-\d{2}[\sT]\d{2}:\d{2}:\d{2})', line)
        if timestamp_match:
            return timestamp_match.group(1)
        return None
    
    def get_recent_errors(self, log_file: str, hours: int = 24, limit: int = 50) -> List[Dict]:
        """获取最近的错误日志"""
        cutoff_time = datetime.now() - timedelta(hours=hours)
        log_path = self.log_dir / log_file
        
        if not log_path.exists():
            return []
        
        errors = []
        
        try:
            with open(log_path, 'r', encoding='utf-8') as f:
                for line_num, line in enumerate(f, 1):
                    if 'ERROR' in line:
                        timestamp_str = self._extract_timestamp(line)
                        if timestamp_str:
                            try:
                                timestamp = datetime.strptime(timestamp_str, '%Y-%m-%d %H:%M:%S')
                                if timestamp >= cutoff_time:
                                    errors.append({
                                        'line': line_num,
                                        'timestamp': timestamp_str,
                                        'content': line.strip(),
                                    })
                            except ValueError:
                                pass
                        
                        if len(errors) >= limit:
                            break
        
        except Exception as e:
            logger.error(f"获取错误日志失败: {e}")
        
        return errors
    
    def get_log_files(self) -> List[str]:
        """获取所有日志文件"""
        log_files = []
        for log_file in self.log_dir.glob("*.log"):
            log_files.append(log_file.name)
        return sorted(log_files)

