"""
SSL证书管理器
"""
import os
import subprocess
import json
from typing import Optional, Dict
from datetime import datetime, timedelta
from pathlib import Path
from loguru import logger
import aiohttp
import base64

from common.models.domain import Domain


class SSLManager:
    """SSL证书管理器"""
    
    def __init__(self, cert_dir: str = "certs"):
        self.cert_dir = Path(cert_dir)
        self.cert_dir.mkdir(parents=True, exist_ok=True)
        self.acme_email = os.getenv("ACME_EMAIL", "admin@example.com")
    
    def get_cert_path(self, domain: str) -> tuple[Path, Path]:
        """获取证书路径"""
        cert_file = self.cert_dir / f"{domain}.crt"
        key_file = self.cert_dir / f"{domain}.key"
        return cert_file, key_file
    
    async def request_certificate(self, domain: Domain) -> bool:
        """申请SSL证书"""
        try:
            cert_file, key_file = self.get_cert_path(domain.domain)
            
            logger.info(f"申请SSL证书: {domain.domain}")
            
            # 方法1: 使用certbot（如果可用）
            if self._certbot_available():
                return await self._request_with_certbot(domain, cert_file, key_file)
            
            # 方法2: 使用ACME客户端库（需要实现）
            # return await self._request_with_acme_client(domain, cert_file, key_file)
            
            # 临时：使用自签名证书（仅用于测试）
            logger.warning("certbot不可用，生成自签名证书（仅用于测试）")
            return await self._generate_self_signed_cert(domain, cert_file, key_file)
            
        except Exception as e:
            logger.error(f"申请SSL证书失败: {e}")
            return False
    
    def _certbot_available(self) -> bool:
        """检查certbot是否可用"""
        try:
            result = subprocess.run(['certbot', '--version'], capture_output=True, timeout=5)
            return result.returncode == 0
        except:
            return False
    
    async def _request_with_certbot(self, domain: Domain, cert_file: Path, key_file: Path) -> bool:
        """使用certbot申请证书"""
        try:
            # 使用standalone模式申请证书
            cmd = [
                'certbot', 'certonly',
                '--standalone',
                '--non-interactive',
                '--agree-tos',
                '--email', self.acme_email,
                '-d', domain.domain,
                '--cert-path', str(cert_file.parent),
                '--key-path', str(key_file.parent),
            ]
            
            result = subprocess.run(cmd, capture_output=True, text=True, timeout=300)
            
            if result.returncode == 0:
                # certbot默认保存路径
                certbot_cert = Path(f'/etc/letsencrypt/live/{domain.domain}/fullchain.pem')
                certbot_key = Path(f'/etc/letsencrypt/live/{domain.domain}/privkey.pem')
                
                if certbot_cert.exists() and certbot_key.exists():
                    # 复制到项目目录
                    import shutil
                    shutil.copy(certbot_cert, cert_file)
                    shutil.copy(certbot_key, key_file)
                    
                    domain.ssl_cert_path = str(cert_file)
                    domain.ssl_key_path = str(key_file)
                    domain.ssl_enabled = True
                    logger.info(f"SSL证书申请成功: {domain.domain}")
                    return True
            
            logger.error(f"certbot申请失败: {result.stderr}")
            return False
            
        except subprocess.TimeoutExpired:
            logger.error("certbot申请超时")
            return False
        except Exception as e:
            logger.error(f"certbot申请异常: {e}")
            return False
    
    async def _generate_self_signed_cert(self, domain: Domain, cert_file: Path, key_file: Path) -> bool:
        """生成自签名证书（仅用于测试）"""
        try:
            # 使用openssl生成自签名证书
            cmd = [
                'openssl', 'req', '-x509', '-newkey', 'rsa:4096',
                '-keyout', str(key_file),
                '-out', str(cert_file),
                '-days', '365',
                '-nodes',
                '-subj', f'/CN={domain.domain}',
            ]
            
            result = subprocess.run(
                cmd,
                input=b'\n',  # 自动确认
                capture_output=True,
                timeout=30
            )
            
            if result.returncode == 0 and cert_file.exists() and key_file.exists():
                domain.ssl_cert_path = str(cert_file)
                domain.ssl_key_path = str(key_file)
                domain.ssl_enabled = True
                logger.info(f"自签名证书生成成功: {domain.domain}")
                return True
            
            return False
            
        except Exception as e:
            logger.error(f"生成自签名证书失败: {e}")
            return False
    
    async def _request_with_acme_client(self, domain: Domain, cert_file: Path, key_file: Path) -> bool:
        """使用ACME客户端库申请证书（需要实现）"""
        # TODO: 实现ACME协议客户端
        # 需要使用acme库或certbot-acme库
        # 实现HTTP-01或DNS-01挑战
        logger.warning("ACME客户端实现待完成")
        return False
    
    def check_certificate_expiry(self, domain: Domain) -> Optional[datetime]:
        """检查证书过期时间"""
        if not domain.ssl_cert_path or not Path(domain.ssl_cert_path).exists():
            return None
        
        try:
            # 使用openssl检查证书过期时间
            result = subprocess.run(
                ['openssl', 'x509', '-noout', '-enddate', '-in', domain.ssl_cert_path],
                capture_output=True,
                text=True,
                timeout=10
            )
            if result.returncode == 0:
                # 解析过期时间
                # 输出格式: notAfter=Dec 31 23:59:59 2024 GMT
                output = result.stdout.strip()
                if 'notAfter=' in output:
                    date_str = output.split('notAfter=')[1].strip()
                    try:
                        # 解析日期
                        expiry = datetime.strptime(date_str, '%b %d %H:%M:%S %Y %Z')
                        return expiry
                    except ValueError:
                        logger.warning(f"无法解析证书过期时间: {date_str}")
        except subprocess.TimeoutExpired:
            logger.error("检查证书过期时间超时")
        except Exception as e:
            logger.error(f"检查证书过期时间失败: {e}")
        
        return None
    
    async def renew_certificate(self, domain: Domain) -> bool:
        """续期SSL证书"""
        expiry = self.check_certificate_expiry(domain)
        if expiry and expiry > datetime.now() + timedelta(days=30):
            logger.info(f"证书尚未到期，无需续期: {domain.domain}")
            return True
        
        logger.info(f"续期SSL证书: {domain.domain}")
        return await self.request_certificate(domain)
    
    def get_certificate_info(self, domain: Domain) -> Optional[dict]:
        """获取证书信息"""
        if not domain.ssl_cert_path or not Path(domain.ssl_cert_path).exists():
            return None
        
        try:
            cert_file = Path(domain.ssl_cert_path)
            key_file = Path(domain.ssl_key_path) if domain.ssl_key_path else None
            
            return {
                'domain': domain.domain,
                'cert_path': str(cert_file),
                'key_path': str(key_file) if key_file else None,
                'cert_exists': cert_file.exists(),
                'key_exists': key_file.exists() if key_file else False,
                'expiry': self.check_certificate_expiry(domain),
            }
        except Exception as e:
            logger.error(f"获取证书信息失败: {e}")
            return None

