"""
域名管理器
"""
import uuid
from typing import Dict, Optional, List
from datetime import datetime, timedelta
from loguru import logger
import dns.resolver
import dns.update
import dns.query

from common.models.domain import Domain, DomainStatus


class DomainManager:
    """域名管理器"""
    
    def __init__(self):
        self._domains: Dict[str, Domain] = {}
        self._client_domains: Dict[str, List[str]] = {}  # client_id -> domain_ids
    
    def create_domain(
        self,
        client_id: str,
        domain: str,
        tunnel_id: Optional[str] = None,
        ssl_enabled: bool = False,
    ) -> Domain:
        """创建域名"""
        domain_obj = Domain(
            id=str(uuid.uuid4()),
            domain=domain,
            client_id=client_id,
            tunnel_id=tunnel_id,
            ssl_enabled=ssl_enabled,
            status=DomainStatus.PENDING,
            created_at=datetime.now(),
            updated_at=datetime.now(),
        )
        
        self._domains[domain_obj.id] = domain_obj
        
        if client_id not in self._client_domains:
            self._client_domains[client_id] = []
        self._client_domains[client_id].append(domain_obj.id)
        
        logger.info(f"创建域名: {domain} (ID: {domain_obj.id})")
        return domain_obj
    
    def get_domain(self, domain_id: str) -> Optional[Domain]:
        """获取域名"""
        return self._domains.get(domain_id)
    
    def get_domain_by_name(self, domain_name: str) -> Optional[Domain]:
        """根据域名名称获取"""
        for domain in self._domains.values():
            if domain.domain == domain_name:
                return domain
        return None
    
    def get_client_domains(self, client_id: str) -> List[Domain]:
        """获取客户端的所有域名"""
        domain_ids = self._client_domains.get(client_id, [])
        return [self._domains[did] for did in domain_ids if did in self._domains]
    
    def update_domain_status(self, domain_id: str, status: DomainStatus):
        """更新域名状态"""
        if domain_id in self._domains:
            domain = self._domains[domain_id]
            domain.status = status
            domain.updated_at = datetime.now()
            logger.info(f"更新域名状态: {domain.domain} -> {status.value}")
    
    def verify_domain(self, domain_id: str) -> bool:
        """验证域名"""
        domain = self.get_domain(domain_id)
        if not domain:
            return False
        
        try:
            # 检查DNS记录
            answers = dns.resolver.resolve(domain.domain, 'A')
            if answers:
                self.update_domain_status(domain_id, DomainStatus.ACTIVE)
                return True
            else:
                self.update_domain_status(domain_id, DomainStatus.ERROR)
                return False
        except Exception as e:
            logger.error(f"验证域名失败: {e}")
            self.update_domain_status(domain_id, DomainStatus.ERROR)
            return False
    
    async def update_dns_record(self, domain_id: str, ip_address: str) -> bool:
        """更新DNS记录"""
        domain = self.get_domain(domain_id)
        if not domain:
            return False
        
        try:
            # 这里应该调用DNS提供商的API
            # 简化实现，实际需要集成Route53、DNSPod等
            logger.info(f"更新DNS记录: {domain.domain} -> {ip_address}")
            # TODO: 实现DNS记录更新
            return True
        except Exception as e:
            logger.error(f"更新DNS记录失败: {e}")
            return False

