"""
ACME客户端 - Let's Encrypt证书申请
"""
import asyncio
import aiohttp
import json
import base64
from typing import Optional, Dict
from pathlib import Path
from loguru import logger
from cryptography import x509
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.x509.oid import NameOID
import josepy as jose


class ACMEClient:
    """ACME客户端（简化实现）"""
    
    # Let's Encrypt目录URL
    DIRECTORY_URL = "https://acme-v02.api.letsencrypt.org/directory"
    STAGING_DIRECTORY_URL = "https://acme-staging-v02.api.letsencrypt.org/directory"
    
    def __init__(self, email: str, staging: bool = False):
        self.email = email
        self.staging = staging
        self.directory_url = self.STAGING_DIRECTORY_URL if staging else self.DIRECTORY_URL
        self.session: Optional[aiohttp.ClientSession] = None
        self.account_key = None
        self.account_url = None
    
    async def __aenter__(self):
        self.session = aiohttp.ClientSession()
        await self._initialize()
        return self
    
    async def __aexit__(self, exc_type, exc_val, exc_tb):
        if self.session:
            await self.session.close()
    
    async def _initialize(self):
        """初始化ACME客户端"""
        try:
            # 获取目录
            async with self.session.get(self.directory_url) as resp:
                self.directory = await resp.json()
            
            # 生成账户密钥
            self.account_key = rsa.generate_private_key(
                public_exponent=65537,
                key_size=2048
            )
            
            # 注册账户
            await self._register_account()
            
        except Exception as e:
            logger.error(f"ACME客户端初始化失败: {e}")
            raise
    
    async def _register_account(self):
        """注册ACME账户"""
        # TODO: 实现账户注册
        logger.info("ACME账户注册（待实现）")
    
    async def request_certificate(self, domain: str, challenge_type: str = "http-01") -> Optional[Dict]:
        """申请证书"""
        try:
            # 1. 创建订单
            order = await self._create_order(domain)
            
            # 2. 完成挑战
            await self._complete_challenge(order, domain, challenge_type)
            
            # 3. 生成CSR
            csr = self._generate_csr(domain)
            
            # 4. 最终化订单
            certificate_url = await self._finalize_order(order, csr)
            
            # 5. 下载证书
            certificate = await self._download_certificate(certificate_url)
            
            return certificate
            
        except Exception as e:
            logger.error(f"申请证书失败: {e}")
            return None
    
    async def _create_order(self, domain: str) -> Dict:
        """创建订单"""
        # TODO: 实现订单创建
        logger.info(f"创建订单: {domain}")
        return {}
    
    async def _complete_challenge(self, order: Dict, domain: str, challenge_type: str):
        """完成挑战"""
        # TODO: 实现HTTP-01或DNS-01挑战
        logger.info(f"完成挑战: {domain} ({challenge_type})")
    
    def _generate_csr(self, domain: str) -> bytes:
        """生成CSR"""
        # 生成私钥
        private_key = rsa.generate_private_key(
            public_exponent=65537,
            key_size=2048
        )
        
        # 创建CSR
        csr = x509.CertificateSigningRequestBuilder().subject_name(
            x509.Name([x509.NameAttribute(NameOID.COMMON_NAME, domain)])
        ).sign(private_key, hashes.SHA256())
        
        return csr.public_bytes(serialization.Encoding.PEM)
    
    async def _finalize_order(self, order: Dict, csr: bytes) -> str:
        """最终化订单"""
        # TODO: 实现订单最终化
        return ""
    
    async def _download_certificate(self, certificate_url: str) -> Dict:
        """下载证书"""
        # TODO: 实现证书下载
        return {}


# 使用示例（需要在HTTP服务器中实现挑战验证）
async def request_certificate_with_acme(domain: str, email: str, webroot: str) -> bool:
    """使用ACME客户端申请证书"""
    async with ACMEClient(email=email, staging=True) as client:
        result = await client.request_certificate(domain)
        if result:
            # 保存证书和密钥
            cert_file = Path(webroot) / f"{domain}.crt"
            key_file = Path(webroot) / f"{domain}.key"
            # TODO: 保存证书和密钥
            return True
    return False

