"""
配置热重载
"""
import asyncio
import yaml
from pathlib import Path
from typing import Dict, Callable, Optional
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler
from loguru import logger


class ConfigReloader:
    """配置热重载器"""
    
    def __init__(self, config_file: str = "config.yaml"):
        self.config_file = Path(config_file)
        self._config: Dict = {}
        self._callbacks: List[Callable] = []
        self._observer: Optional[Observer] = None
        self._running = False
    
    def add_reload_callback(self, callback: Callable):
        """添加重载回调"""
        self._callbacks.append(callback)
    
    def load_config(self) -> Dict:
        """加载配置"""
        if not self.config_file.exists():
            logger.warning(f"配置文件不存在: {self.config_file}")
            return {}
        
        try:
            with open(self.config_file, 'r', encoding='utf-8') as f:
                self._config = yaml.safe_load(f) or {}
            logger.info(f"配置已加载: {self.config_file}")
            return self._config
        except Exception as e:
            logger.error(f"加载配置失败: {e}")
            return {}
    
    def get_config(self, key: str, default=None):
        """获取配置值"""
        keys = key.split('.')
        value = self._config
        for k in keys:
            if isinstance(value, dict):
                value = value.get(k)
                if value is None:
                    return default
            else:
                return default
        return value
    
    def _on_config_changed(self):
        """配置文件变更处理"""
        logger.info("检测到配置文件变更，重新加载...")
        old_config = self._config.copy()
        new_config = self.load_config()
        
        # 通知所有回调
        for callback in self._callbacks:
            try:
                callback(old_config, new_config)
            except Exception as e:
                logger.error(f"配置重载回调执行失败: {e}")
    
    def start_watching(self):
        """开始监控配置文件"""
        if self._running:
            return
        
        if not self.config_file.exists():
            logger.warning(f"配置文件不存在，无法监控: {self.config_file}")
            return
        
        class ConfigFileHandler(FileSystemEventHandler):
            def __init__(self, reloader):
                self.reloader = reloader
            
            def on_modified(self, event):
                if event.src_path == str(self.reloader.config_file):
                    self.reloader._on_config_changed()
        
        self._observer = Observer()
        self._observer.schedule(
            ConfigFileHandler(self),
            str(self.config_file.parent),
            recursive=False
        )
        self._observer.start()
        self._running = True
        logger.info(f"开始监控配置文件: {self.config_file}")
    
    def stop_watching(self):
        """停止监控"""
        if self._observer:
            self._observer.stop()
            self._observer.join()
            self._running = False
            logger.info("停止监控配置文件")

