"""
会话同步 - Redis集群会话同步
"""
import json
import asyncio
from typing import Dict, Optional
from datetime import datetime
from loguru import logger
try:
    import redis.asyncio as aioredis
except ImportError:
    import aioredis


class SessionSync:
    """会话同步管理器"""
    
    def __init__(self, redis_url: str = "redis://localhost:6379/0"):
        self.redis_url = redis_url
        self.redis: Optional[aioredis.Redis] = None
        self._pubsub = None
        self._subscribed = False
        self._channel = "nps:session:sync"
    
    async def connect(self):
        """连接Redis"""
        try:
            self.redis = await aioredis.from_url(self.redis_url)
            await self.redis.ping()
            logger.info("Redis连接成功")
        except Exception as e:
            logger.error(f"Redis连接失败: {e}")
            self.redis = None
    
    async def disconnect(self):
        """断开Redis连接"""
        if self._pubsub:
            await self._pubsub.unsubscribe()
            await self._pubsub.close()
            self._pubsub = None
        
        if self.redis:
            await self.redis.close()
            self.redis = None
    
    async def sync_session(self, session_id: str, session_data: Dict):
        """同步会话数据"""
        if not self.redis:
            return
        
        try:
            key = f"session:{session_id}"
            await self.redis.setex(
                key,
                3600,  # 1小时过期
                json.dumps(session_data)
            )
            
            # 发布同步消息
            await self.redis.publish(
                self._channel,
                json.dumps({
                    'action': 'sync',
                    'session_id': session_id,
                    'data': session_data,
                })
            )
        except Exception as e:
            logger.error(f"同步会话失败: {e}")
    
    async def get_session(self, session_id: str) -> Optional[Dict]:
        """获取会话数据"""
        if not self.redis:
            return None
        
        try:
            key = f"session:{session_id}"
            data = await self.redis.get(key)
            if data:
                return json.loads(data)
        except Exception as e:
            logger.error(f"获取会话失败: {e}")
        
        return None
    
    async def delete_session(self, session_id: str):
        """删除会话"""
        if not self.redis:
            return
        
        try:
            key = f"session:{session_id}"
            await self.redis.delete(key)
            
            # 发布删除消息
            await self.redis.publish(
                self._channel,
                json.dumps({
                    'action': 'delete',
                    'session_id': session_id,
                })
            )
        except Exception as e:
            logger.error(f"删除会话失败: {e}")
    
    async def subscribe_updates(self, callback):
        """订阅会话更新"""
        if not self.redis or self._subscribed:
            return
        
        try:
            self._pubsub = self.redis.pubsub()
            await self._pubsub.subscribe(self._channel)
            self._subscribed = True
            
            # 启动监听循环
            asyncio.create_task(self._listen_updates(callback))
            logger.info("已订阅会话更新")
        except Exception as e:
            logger.error(f"订阅会话更新失败: {e}")
    
    async def _listen_updates(self, callback):
        """监听更新"""
        try:
            while self._subscribed and self._pubsub:
                message = await self._pubsub.get_message(ignore_subscribe_messages=True, timeout=1.0)
                if message:
                    try:
                        data = json.loads(message['data'])
                        await callback(data)
                    except Exception as e:
                        logger.error(f"处理会话更新消息失败: {e}")
        except Exception as e:
            logger.error(f"监听会话更新失败: {e}")

