"""
缓存策略优化
"""
from typing import Any, Optional, Callable
from enum import Enum
from datetime import datetime, timedelta
from loguru import logger
import hashlib
import json


class CacheStrategy(str, Enum):
    """缓存策略"""
    LRU = "lru"  # 最近最少使用
    LFU = "lfu"  # 最少使用频率
    FIFO = "fifo"  # 先进先出
    TTL = "ttl"  # 基于时间过期


class CacheStrategyManager:
    """缓存策略管理器"""
    
    def __init__(self, strategy: CacheStrategy = CacheStrategy.LRU):
        self.strategy = strategy
        self._cache: Dict[str, Any] = {}
        self._access_times: Dict[str, datetime] = {}
        self._access_counts: Dict[str, int] = {}
        self._max_size = 1000
    
    def set_max_size(self, size: int):
        """设置最大缓存大小"""
        self._max_size = size
    
    def _generate_key(self, *args, **kwargs) -> str:
        """生成缓存键"""
        key_data = {
            'args': args,
            'kwargs': kwargs
        }
        key_str = json.dumps(key_data, sort_keys=True, default=str)
        return hashlib.md5(key_str.encode()).hexdigest()
    
    def get(self, key: str) -> Optional[Any]:
        """获取缓存"""
        if key not in self._cache:
            return None
        
        # 更新访问时间和计数
        self._access_times[key] = datetime.now()
        self._access_counts[key] = self._access_counts.get(key, 0) + 1
        
        return self._cache[key]
    
    def set(self, key: str, value: Any, ttl: Optional[int] = None):
        """设置缓存"""
        # 检查是否需要清理
        if len(self._cache) >= self._max_size:
            self._evict()
        
        self._cache[key] = {
            'value': value,
            'created_at': datetime.now(),
            'ttl': ttl,
        }
        self._access_times[key] = datetime.now()
        self._access_counts[key] = 0
    
    def _evict(self):
        """淘汰缓存项"""
        if self.strategy == CacheStrategy.LRU:
            # 移除最近最少使用的
            if self._access_times:
                lru_key = min(self._access_times.items(), key=lambda x: x[1])[0]
                self._remove(lru_key)
        
        elif self.strategy == CacheStrategy.LFU:
            # 移除最少使用频率的
            if self._access_counts:
                lfu_key = min(self._access_counts.items(), key=lambda x: x[1])[0]
                self._remove(lfu_key)
        
        elif self.strategy == CacheStrategy.FIFO:
            # 移除最早创建的
            if self._cache:
                oldest_key = min(
                    self._cache.items(),
                    key=lambda x: x[1]['created_at']
                )[0]
                self._remove(oldest_key)
    
    def _remove(self, key: str):
        """移除缓存项"""
        if key in self._cache:
            del self._cache[key]
        if key in self._access_times:
            del self._access_times[key]
        if key in self._access_counts:
            del self._access_counts[key]
    
    def _check_ttl(self, key: str) -> bool:
        """检查TTL"""
        if key not in self._cache:
            return False
        
        item = self._cache[key]
        if item['ttl']:
            if datetime.now() - item['created_at'] > timedelta(seconds=item['ttl']):
                self._remove(key)
                return False
        
        return True
    
    def clear(self):
        """清空缓存"""
        self._cache.clear()
        self._access_times.clear()
        self._access_counts.clear()
    
    def get_stats(self) -> Dict:
        """获取统计信息"""
        # 清理过期项
        for key in list(self._cache.keys()):
            self._check_ttl(key)
        
        return {
            'size': len(self._cache),
            'max_size': self._max_size,
            'strategy': self.strategy.value,
            'hit_rate': self._calculate_hit_rate(),
        }
    
    def _calculate_hit_rate(self) -> float:
        """计算命中率"""
        total_access = sum(self._access_counts.values())
        if total_access == 0:
            return 0.0
        hits = len([c for c in self._access_counts.values() if c > 0])
        return hits / total_access if total_access > 0 else 0.0


def cache_result(strategy: CacheStrategy = CacheStrategy.LRU, ttl: Optional[int] = None):
    """缓存结果装饰器"""
    cache_manager = CacheStrategyManager(strategy)
    
    def decorator(func: Callable):
        async def async_wrapper(*args, **kwargs):
            key = cache_manager._generate_key(*args, **kwargs)
            
            # 检查缓存
            cached = cache_manager.get(key)
            if cached:
                return cached['value']
            
            # 执行函数
            result = await func(*args, **kwargs)
            
            # 缓存结果
            cache_manager.set(key, result, ttl)
            
            return result
        
        def sync_wrapper(*args, **kwargs):
            key = cache_manager._generate_key(*args, **kwargs)
            
            # 检查缓存
            cached = cache_manager.get(key)
            if cached and cache_manager._check_ttl(key):
                return cached['value']
            
            # 执行函数
            result = func(*args, **kwargs)
            
            # 缓存结果
            cache_manager.set(key, result, ttl)
            
            return result
        
        import asyncio
        if asyncio.iscoroutinefunction(func):
            return async_wrapper
        else:
            return sync_wrapper
    
    return decorator

