"""
缓存管理器
"""
import time
from typing import Dict, Optional, Any, Callable
from datetime import datetime, timedelta
from loguru import logger
import json
import hashlib


class CacheManager:
    """缓存管理器"""
    
    def __init__(self, default_ttl: int = 300):
        """
        初始化缓存管理器
        
        Args:
            default_ttl: 默认过期时间（秒）
        """
        self._cache: Dict[str, tuple[Any, float]] = {}  # key -> (value, expire_time)
        self.default_ttl = default_ttl
    
    def get(self, key: str) -> Optional[Any]:
        """获取缓存值"""
        if key not in self._cache:
            return None
        
        value, expire_time = self._cache[key]
        
        # 检查是否过期
        if time.time() > expire_time:
            del self._cache[key]
            return None
        
        return value
    
    def set(self, key: str, value: Any, ttl: Optional[int] = None):
        """设置缓存值"""
        ttl = ttl or self.default_ttl
        expire_time = time.time() + ttl
        self._cache[key] = (value, expire_time)
    
    def delete(self, key: str):
        """删除缓存"""
        if key in self._cache:
            del self._cache[key]
    
    def clear(self):
        """清空所有缓存"""
        self._cache.clear()
    
    def get_or_set(self, key: str, func: Callable, ttl: Optional[int] = None) -> Any:
        """获取缓存值，如果不存在则调用函数设置"""
        value = self.get(key)
        if value is None:
            value = func()
            self.set(key, value, ttl)
        return value
    
    async def get_or_set_async(self, key: str, func: Callable, ttl: Optional[int] = None) -> Any:
        """异步获取缓存值，如果不存在则调用异步函数设置"""
        value = self.get(key)
        if value is None:
            value = await func()
            self.set(key, value, ttl)
        return value
    
    def cleanup_expired(self):
        """清理过期缓存"""
        current_time = time.time()
        expired_keys = [
            key for key, (_, expire_time) in self._cache.items()
            if current_time > expire_time
        ]
        for key in expired_keys:
            del self._cache[key]
        
        if expired_keys:
            logger.debug(f"清理了 {len(expired_keys)} 个过期缓存")
    
    def get_stats(self) -> Dict[str, Any]:
        """获取缓存统计"""
        current_time = time.time()
        total = len(self._cache)
        expired = sum(1 for _, expire_time in self._cache.values() if current_time > expire_time)
        active = total - expired
        
        return {
            'total': total,
            'active': active,
            'expired': expired,
        }
    
    @staticmethod
    def make_key(*args, **kwargs) -> str:
        """生成缓存键"""
        key_data = {
            'args': args,
            'kwargs': sorted(kwargs.items())
        }
        key_str = json.dumps(key_data, sort_keys=True)
        return hashlib.md5(key_str.encode()).hexdigest()

