"""
备份管理器（支持加密）
"""
import os
import shutil
import json
import subprocess
from pathlib import Path
from typing import List, Dict, Optional
from datetime import datetime
from loguru import logger
from cryptography.fernet import Fernet
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
import base64


class BackupManager:
    """备份管理器（支持加密）"""
    
    def __init__(self, backup_dir: str = "backups", encryption_key: str = None):
        self.backup_dir = Path(backup_dir)
        self.backup_dir.mkdir(parents=True, exist_ok=True)
        self.encryption_enabled = False
        self.cipher_suite = None
        
        if encryption_key:
            self.enable_encryption(encryption_key)
    
    def enable_encryption(self, password: str):
        """启用备份加密"""
        try:
            # 从密码生成密钥
            kdf = PBKDF2HMAC(
                algorithm=hashes.SHA256(),
                length=32,
                salt=b'nps_backup_salt',  # 在生产环境中应该使用随机salt
                iterations=100000,
            )
            key = base64.urlsafe_b64encode(kdf.derive(password.encode()))
            self.cipher_suite = Fernet(key)
            self.encryption_enabled = True
            logger.info("备份加密已启用")
        except Exception as e:
            logger.error(f"启用备份加密失败: {e}")
            self.encryption_enabled = False
    
    def _encrypt_data(self, data: bytes) -> bytes:
        """加密数据"""
        if self.encryption_enabled and self.cipher_suite:
            return self.cipher_suite.encrypt(data)
        return data
    
    def _decrypt_data(self, encrypted_data: bytes) -> bytes:
        """解密数据"""
        if self.encryption_enabled and self.cipher_suite:
            return self.cipher_suite.decrypt(encrypted_data)
        return encrypted_data
    
    def create_backup(self, name: str = None, encrypt: bool = None) -> str:
        """创建备份"""
        if name is None:
            name = f"backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        
        if encrypt is None:
            encrypt = self.encryption_enabled
        
        backup_path = self.backup_dir / name
        backup_path.mkdir(parents=True, exist_ok=True)
        
        try:
            # 备份配置文件
            config_files = [
                "config.yaml",
                ".env",
            ]
            
            for config_file in config_files:
                src = Path(config_file)
                if src.exists():
                    dst = backup_path / config_file
                    dst.parent.mkdir(parents=True, exist_ok=True)
                    shutil.copy2(src, dst)
            
            # 备份证书
            certs_dir = Path("certs")
            if certs_dir.exists():
                dst_certs = backup_path / "certs"
                shutil.copytree(certs_dir, dst_certs, dirs_exist_ok=True)
            
            # 备份数据库（如果配置了）
            database_url = os.getenv("DATABASE_URL")
            if database_url and "postgresql" in database_url:
                self._backup_database(backup_path, database_url)
            
            # 创建备份元数据
            metadata = {
                'name': name,
                'created_at': datetime.now().isoformat(),
                'type': 'full',
                'encrypted': encrypt,
            }
            
            metadata_file = backup_path / "metadata.json"
            metadata_data = json.dumps(metadata, indent=2).encode('utf-8')
            
            if encrypt:
                # 加密元数据
                encrypted_metadata = self._encrypt_data(metadata_data)
                with open(metadata_file.with_suffix('.json.encrypted'), 'wb') as f:
                    f.write(encrypted_metadata)
            else:
                with open(metadata_file, 'w') as f:
                    json.dump(metadata, f, indent=2)
            
            logger.info(f"备份创建成功: {backup_path}")
            return str(backup_path)
            
        except Exception as e:
            logger.error(f"创建备份失败: {e}")
            raise
    
    def _backup_database(self, backup_path: Path, database_url: str):
        """备份数据库"""
        try:
            # 解析数据库URL
            # postgresql://user:password@host:port/database
            from urllib.parse import urlparse
            parsed = urlparse(database_url)
            
            db_name = parsed.path.lstrip('/')
            db_host = parsed.hostname
            db_port = parsed.port or 5432
            db_user = parsed.username
            db_password = parsed.password
            
            # 使用pg_dump备份
            dump_file = backup_path / "database.sql"
            
            env = os.environ.copy()
            if db_password:
                env['PGPASSWORD'] = db_password
            
            cmd = [
                'pg_dump',
                '-h', db_host,
                '-p', str(db_port),
                '-U', db_user,
                '-d', db_name,
                '-f', str(dump_file),
            ]
            
            result = subprocess.run(cmd, env=env, capture_output=True, text=True)
            
            if result.returncode == 0:
                # 如果启用加密，加密数据库备份文件
                if self.encryption_enabled:
                    with open(dump_file, 'rb') as f:
                        data = f.read()
                    encrypted_data = self._encrypt_data(data)
                    encrypted_file = backup_path / "database.sql.encrypted"
                    with open(encrypted_file, 'wb') as f:
                        f.write(encrypted_data)
                    dump_file.unlink()  # 删除未加密文件
                    logger.info(f"数据库备份已加密: {encrypted_file}")
                else:
                    logger.info(f"数据库备份成功: {dump_file}")
            else:
                logger.error(f"数据库备份失败: {result.stderr}")
                
        except Exception as e:
            logger.error(f"数据库备份异常: {e}")
    
    def list_backups(self) -> List[Dict]:
        """列出所有备份"""
        backups = []
        
        for backup_dir in self.backup_dir.iterdir():
            if not backup_dir.is_dir():
                continue
            
            metadata_file = backup_dir / "metadata.json"
            if metadata_file.exists():
                try:
                    with open(metadata_file, 'r') as f:
                        metadata = json.load(f)
                    backups.append({
                        'name': metadata.get('name', backup_dir.name),
                        'created_at': metadata.get('created_at'),
                        'type': metadata.get('type', 'unknown'),
                        'path': str(backup_dir),
                    })
                except Exception as e:
                    logger.error(f"读取备份元数据失败: {e}")
        
        return sorted(backups, key=lambda x: x.get('created_at', ''), reverse=True)
    
    def restore_backup(self, backup_name: str) -> bool:
        """恢复备份"""
        backup_path = self.backup_dir / backup_name
        
        if not backup_path.exists():
            logger.error(f"备份不存在: {backup_name}")
            return False
        
        try:
            # 恢复配置文件
            for config_file in backup_path.glob("*.yaml"):
                dst = Path(config_file.name)
                shutil.copy2(config_file, dst)
                logger.info(f"恢复配置文件: {config_file.name}")
            
            for config_file in backup_path.glob(".env"):
                dst = Path(".env")
                shutil.copy2(config_file, dst)
                logger.info("恢复环境配置文件")
            
            # 恢复证书
            certs_backup = backup_path / "certs"
            if certs_backup.exists():
                certs_dir = Path("certs")
                if certs_dir.exists():
                    shutil.rmtree(certs_dir)
                shutil.copytree(certs_backup, certs_dir)
                logger.info("恢复证书文件")
            
            # 恢复数据库
            dump_file = backup_path / "database.sql"
            encrypted_dump_file = backup_path / "database.sql.encrypted"
            
            if encrypted_dump_file.exists():
                # 解密数据库备份
                with open(encrypted_dump_file, 'rb') as f:
                    encrypted_data = f.read()
                decrypted_data = self._decrypt_data(encrypted_data)
                with open(dump_file, 'wb') as f:
                    f.write(decrypted_data)
                self._restore_database(dump_file)
                dump_file.unlink()  # 删除临时解密文件
            elif dump_file.exists():
                self._restore_database(dump_file)
            
            logger.info(f"备份恢复成功: {backup_name}")
            return True
            
        except Exception as e:
            logger.error(f"恢复备份失败: {e}")
            return False
    
    def _restore_database(self, dump_file: Path):
        """恢复数据库"""
        try:
            database_url = os.getenv("DATABASE_URL")
            if not database_url or "postgresql" not in database_url:
                logger.warning("未配置数据库，跳过数据库恢复")
                return
            
            from urllib.parse import urlparse
            parsed = urlparse(database_url)
            
            db_name = parsed.path.lstrip('/')
            db_host = parsed.hostname
            db_port = parsed.port or 5432
            db_user = parsed.username
            db_password = parsed.password
            
            env = os.environ.copy()
            if db_password:
                env['PGPASSWORD'] = db_password
            
            cmd = [
                'psql',
                '-h', db_host,
                '-p', str(db_port),
                '-U', db_user,
                '-d', db_name,
                '-f', str(dump_file),
            ]
            
            result = subprocess.run(cmd, env=env, capture_output=True, text=True)
            
            if result.returncode == 0:
                logger.info("数据库恢复成功")
            else:
                logger.error(f"数据库恢复失败: {result.stderr}")
                
        except Exception as e:
            logger.error(f"数据库恢复异常: {e}")
    
    def delete_backup(self, backup_name: str) -> bool:
        """删除备份"""
        backup_path = self.backup_dir / backup_name
        
        if not backup_path.exists():
            logger.error(f"备份不存在: {backup_name}")
            return False
        
        try:
            shutil.rmtree(backup_path)
            logger.info(f"备份已删除: {backup_name}")
            return True
        except Exception as e:
            logger.error(f"删除备份失败: {e}")
            return False

