"""
认证管理器
"""
import jwt
from datetime import datetime, timedelta
from typing import Optional, Dict
from loguru import logger

from common.models.client import Client, ClientStatus
from common.utils.crypto import generate_token


class AuthManager:
    """认证管理器"""
    
    def __init__(self, secret_key: str):
        self.secret_key = secret_key
        self.algorithm = "HS256"
        self.token_expire_hours = 24 * 7  # 7天
    
    def verify_token(self, token: str) -> Optional[Dict]:
        """验证token"""
        try:
            payload = jwt.decode(token, self.secret_key, algorithms=[self.algorithm])
            return payload
        except jwt.ExpiredSignatureError:
            logger.warning("Token已过期")
            return None
        except jwt.InvalidTokenError:
            logger.warning("Token无效")
            return None
    
    def generate_token(self, client_id: str, client_name: str) -> str:
        """生成JWT token"""
        payload = {
            'client_id': client_id,
            'client_name': client_name,
            'exp': datetime.utcnow() + timedelta(hours=self.token_expire_hours),
            'iat': datetime.utcnow(),
        }
        return jwt.encode(payload, self.secret_key, algorithm=self.algorithm)
    
    async def authenticate_client(self, token: str, client_id: str) -> Optional[Client]:
        """认证客户端"""
        # 验证token
        payload = self.verify_token(token)
        if not payload:
            return None
        
        # 验证client_id
        if payload.get('client_id') != client_id:
            logger.warning(f"Token中的client_id不匹配: {payload.get('client_id')} != {client_id}")
            return None
        
        # 这里应该从数据库查询客户端信息
        # 暂时返回模拟数据
        client = Client(
            id=client_id,
            name=payload.get('client_name', 'Unknown'),
            token=token,
            status=ClientStatus.ONLINE,
            created_at=datetime.now(),
            updated_at=datetime.now(),
        )
        
        logger.info(f"客户端 {client_id} 认证成功")
        return client
    
    def create_client_token(self, client_id: str, client_name: str) -> str:
        """创建客户端token"""
        return self.generate_token(client_id, client_name)

