"""
API认证和权限管理
"""
from typing import Optional, List, Set
from enum import Enum
from fastapi import HTTPException, Security, Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from loguru import logger
import jwt
from datetime import datetime, timedelta


class Permission(str, Enum):
    """权限枚举"""
    # 客户端管理
    CLIENT_VIEW = "client:view"
    CLIENT_CREATE = "client:create"
    CLIENT_EDIT = "client:edit"
    CLIENT_DELETE = "client:delete"
    
    # 隧道管理
    TUNNEL_VIEW = "tunnel:view"
    TUNNEL_CREATE = "tunnel:create"
    TUNNEL_EDIT = "tunnel:edit"
    TUNNEL_DELETE = "tunnel:delete"
    
    # 域名管理
    DOMAIN_VIEW = "domain:view"
    DOMAIN_CREATE = "domain:create"
    DOMAIN_EDIT = "domain:edit"
    DOMAIN_DELETE = "domain:delete"
    
    # 系统管理
    SYSTEM_VIEW = "system:view"
    SYSTEM_CONFIG = "system:config"
    SYSTEM_BACKUP = "system:backup"


class Role(str, Enum):
    """角色枚举"""
    ADMIN = "admin"  # 管理员：所有权限
    OPERATOR = "operator"  # 操作员：查看和操作权限
    VIEWER = "viewer"  # 查看者：只读权限


# 角色权限映射
ROLE_PERMISSIONS = {
    Role.ADMIN: set(Permission),  # 所有权限
    Role.OPERATOR: {
        Permission.CLIENT_VIEW, Permission.CLIENT_CREATE, Permission.CLIENT_EDIT,
        Permission.TUNNEL_VIEW, Permission.TUNNEL_CREATE, Permission.TUNNEL_EDIT,
        Permission.DOMAIN_VIEW, Permission.DOMAIN_CREATE, Permission.DOMAIN_EDIT,
        Permission.SYSTEM_VIEW,
    },
    Role.VIEWER: {
        Permission.CLIENT_VIEW,
        Permission.TUNNEL_VIEW,
        Permission.DOMAIN_VIEW,
        Permission.SYSTEM_VIEW,
    },
}


class APIAuth:
    """API认证管理器"""
    
    def __init__(self, secret_key: str):
        self.secret_key = secret_key
        self.algorithm = "HS256"
        self.token_expire_hours = 24
        self.security = HTTPBearer()
    
    def create_api_token(self, username: str, role: Role, permissions: Optional[List[Permission]] = None) -> str:
        """创建API Token"""
        if permissions is None:
            permissions = list(ROLE_PERMISSIONS.get(role, set()))
        
        payload = {
            'username': username,
            'role': role.value,
            'permissions': [p.value for p in permissions],
            'exp': datetime.utcnow() + timedelta(hours=self.token_expire_hours),
            'iat': datetime.utcnow(),
        }
        
        return jwt.encode(payload, self.secret_key, algorithm=self.algorithm)
    
    def verify_token(self, token: str) -> Optional[Dict]:
        """验证Token"""
        try:
            payload = jwt.decode(token, self.secret_key, algorithms=[self.algorithm])
            return payload
        except jwt.ExpiredSignatureError:
            logger.warning("API Token已过期")
            return None
        except jwt.InvalidTokenError:
            logger.warning("API Token无效")
            return None
    
    def get_current_user(self, credentials: HTTPAuthorizationCredentials = Security(HTTPBearer())) -> Dict:
        """获取当前用户信息"""
        token = credentials.credentials
        payload = self.verify_token(token)
        
        if not payload:
            raise HTTPException(status_code=401, detail="无效的认证Token")
        
        return {
            'username': payload.get('username'),
            'role': payload.get('role'),
            'permissions': payload.get('permissions', []),
        }
    
    def require_permission(self, permission: Permission):
        """权限检查装饰器"""
        def permission_checker(user: Dict = Depends(lambda: self.get_current_user())):
            user_permissions = set(user.get('permissions', []))
            
            if permission.value not in user_permissions:
                raise HTTPException(
                    status_code=403,
                    detail=f"缺少权限: {permission.value}"
                )
            
            return user
        
        return permission_checker
    
    def require_role(self, *roles: Role):
        """角色检查装饰器"""
        allowed_roles = {r.value for r in roles}
        
        def role_checker(user: Dict = Depends(lambda: self.get_current_user())):
            user_role = user.get('role')
            
            if user_role not in allowed_roles:
                raise HTTPException(
                    status_code=403,
                    detail=f"需要角色: {', '.join(allowed_roles)}"
                )
            
            return user
        
        return role_checker

