"""
告警规则定义
"""
from typing import Dict, Callable, Optional
from dataclasses import dataclass
from enum import Enum
from loguru import logger
from .alert_manager import AlertManager, AlertLevel


class AlertRuleType(str, Enum):
    """告警规则类型"""
    THRESHOLD = "threshold"  # 阈值告警
    RATE_OF_CHANGE = "rate_of_change"  # 变化率告警
    ANOMALY = "anomaly"  # 异常检测告警
    PATTERN = "pattern"  # 模式匹配告警


@dataclass
class AlertRule:
    """告警规则"""
    name: str
    rule_type: AlertRuleType
    metric_name: str
    condition: Callable
    level: AlertLevel
    enabled: bool = True
    cooldown_seconds: int = 300  # 冷却时间（秒）
    last_triggered: Optional[float] = None


class AlertRuleEngine:
    """告警规则引擎"""
    
    def __init__(self, alert_manager: AlertManager):
        self.alert_manager = alert_manager
        self.rules: Dict[str, AlertRule] = {}
        self._initialize_default_rules()
    
    def _initialize_default_rules(self):
        """初始化默认告警规则"""
        import time
        
        # CPU使用率告警
        self.add_rule(AlertRule(
            name="high_cpu_usage",
            rule_type=AlertRuleType.THRESHOLD,
            metric_name="system_cpu_percent",
            condition=lambda value: value > 80.0,
            level=AlertLevel.WARNING,
            cooldown_seconds=300
        ))
        
        # 内存使用率告警
        self.add_rule(AlertRule(
            name="high_memory_usage",
            rule_type=AlertRuleType.THRESHOLD,
            metric_name="system_memory_percent",
            condition=lambda value: value > 85.0,
            level=AlertLevel.WARNING,
            cooldown_seconds=300
        ))
        
        # 磁盘使用率告警
        self.add_rule(AlertRule(
            name="high_disk_usage",
            rule_type=AlertRuleType.THRESHOLD,
            metric_name="system_disk_percent",
            condition=lambda value: value > 90.0,
            level=AlertLevel.ERROR,
            cooldown_seconds=600
        ))
        
        # 连接数告警
        self.add_rule(AlertRule(
            name="high_connection_count",
            rule_type=AlertRuleType.THRESHOLD,
            metric_name="connections_active",
            condition=lambda value: value > 8000,
            level=AlertLevel.WARNING,
            cooldown_seconds=300
        ))
        
        # 错误率告警
        self.add_rule(AlertRule(
            name="high_error_rate",
            rule_type=AlertRuleType.RATE_OF_CHANGE,
            metric_name="errors_total",
            condition=lambda current, previous: (current - previous) > 100,
            level=AlertLevel.ERROR,
            cooldown_seconds=60
        ))
        
        # 响应时间告警
        self.add_rule(AlertRule(
            name="high_response_time",
            rule_type=AlertRuleType.THRESHOLD,
            metric_name="http_request_duration_p95",
            condition=lambda value: value > 2.0,  # 2秒
            level=AlertLevel.WARNING,
            cooldown_seconds=300
        ))
        
        # 限流命中率告警
        self.add_rule(AlertRule(
            name="high_rate_limit_hits",
            rule_type=AlertRuleType.RATE_OF_CHANGE,
            metric_name="rate_limit_hits_total",
            condition=lambda current, previous: (current - previous) > 50,
            level=AlertLevel.WARNING,
            cooldown_seconds=60
        ))
        
        # 熔断器打开告警
        self.add_rule(AlertRule(
            name="circuit_breaker_open",
            rule_type=AlertRuleType.PATTERN,
            metric_name="circuit_breaker_state",
            condition=lambda value: value == 1,  # 1=open
            level=AlertLevel.ERROR,
            cooldown_seconds=60
        ))
        
        # DDoS攻击检测
        self.add_rule(AlertRule(
            name="ddos_attack_detected",
            rule_type=AlertRuleType.ANOMALY,
            metric_name="security_anomalies_total",
            condition=lambda value: value > 0,
            level=AlertLevel.CRITICAL,
            cooldown_seconds=30
        ))
        
        logger.info(f"初始化了 {len(self.rules)} 个告警规则")
    
    def add_rule(self, rule: AlertRule):
        """添加告警规则"""
        self.rules[rule.name] = rule
        logger.info(f"添加告警规则: {rule.name}")
    
    def remove_rule(self, name: str):
        """移除告警规则"""
        if name in self.rules:
            del self.rules[name]
            logger.info(f"移除告警规则: {name}")
    
    def evaluate_rule(self, rule_name: str, metric_value: float, previous_value: float = None):
        """评估告警规则"""
        if rule_name not in self.rules:
            return
        
        rule = self.rules[rule_name]
        if not rule.enabled:
            return
        
        # 检查冷却时间
        import time
        if rule.last_triggered and (time.time() - rule.last_triggered) < rule.cooldown_seconds:
            return
        
        # 根据规则类型评估
        should_alert = False
        
        if rule.rule_type == AlertRuleType.THRESHOLD:
            should_alert = rule.condition(metric_value)
        
        elif rule.rule_type == AlertRuleType.RATE_OF_CHANGE:
            if previous_value is not None:
                should_alert = rule.condition(metric_value, previous_value)
        
        elif rule.rule_type == AlertRuleType.ANOMALY or rule.rule_type == AlertRuleType.PATTERN:
            should_alert = rule.condition(metric_value)
        
        if should_alert:
            self._trigger_alert(rule, metric_value)
            rule.last_triggered = time.time()
    
    def _trigger_alert(self, rule: AlertRule, metric_value: float):
        """触发告警"""
        from datetime import datetime
        
        alert_title = f"告警: {rule.name}"
        alert_message = f"指标 {rule.metric_name} 的值 {metric_value} 触发了告警规则 {rule.name}"
        
        self.alert_manager.send_alert(
            level=rule.level,
            title=alert_title,
            message=alert_message,
            metadata={
                'rule_name': rule.name,
                'metric_name': rule.metric_name,
                'metric_value': metric_value,
                'rule_type': rule.rule_type.value,
            }
        )
        
        logger.warning(f"触发告警: {rule.name} - {alert_message}")
    
    def get_all_rules(self) -> Dict[str, Dict]:
        """获取所有规则"""
        import time
        return {
            name: {
                'name': rule.name,
                'type': rule.rule_type.value,
                'metric_name': rule.metric_name,
                'level': rule.level.value,
                'enabled': rule.enabled,
                'cooldown_seconds': rule.cooldown_seconds,
                'last_triggered': rule.last_triggered,
                'time_since_last_trigger': time.time() - rule.last_triggered if rule.last_triggered else None,
            }
            for name, rule in self.rules.items()
        }
    
    def enable_rule(self, name: str):
        """启用规则"""
        if name in self.rules:
            self.rules[name].enabled = True
            logger.info(f"启用告警规则: {name}")
    
    def disable_rule(self, name: str):
        """禁用规则"""
        if name in self.rules:
            self.rules[name].enabled = False
            logger.info(f"禁用告警规则: {name}")

