"""
告警管理器
"""
import smtplib
import aiohttp
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from typing import List, Dict, Optional, Callable
from dataclasses import dataclass
from datetime import datetime
from enum import Enum
from loguru import logger


class AlertLevel(str, Enum):
    """告警级别"""
    INFO = "info"
    WARNING = "warning"
    ERROR = "error"
    CRITICAL = "critical"


@dataclass
class Alert:
    """告警"""
    level: AlertLevel
    title: str
    message: str
    timestamp: datetime
    metadata: Dict = None


class AlertManager:
    """告警管理器"""
    
    def __init__(self):
        self._alerts: List[Alert] = []
        self._handlers: List[Callable] = []
        self._email_config: Optional[Dict] = None
        self._webhook_urls: List[str] = []
        self._enabled = True
    
    def configure_email(self, smtp_host: str, smtp_port: int, username: str, password: str, from_email: str, to_emails: List[str]):
        """配置邮件通知"""
        self._email_config = {
            'smtp_host': smtp_host,
            'smtp_port': smtp_port,
            'username': username,
            'password': password,
            'from_email': from_email,
            'to_emails': to_emails,
        }
        logger.info("邮件通知配置完成")
    
    def add_webhook(self, url: str):
        """添加Webhook URL"""
        self._webhook_urls.append(url)
        logger.info(f"添加Webhook: {url}")
    
    def add_handler(self, handler: Callable):
        """添加自定义处理器"""
        self._handlers.append(handler)
    
    async def send_alert(self, level: AlertLevel, title: str, message: str, metadata: Dict = None):
        """发送告警"""
        if not self._enabled:
            return
        
        alert = Alert(
            level=level,
            title=title,
            message=message,
            timestamp=datetime.now(),
            metadata=metadata or {}
        )
        
        self._alerts.append(alert)
        
        # 只保留最近1000条告警
        if len(self._alerts) > 1000:
            self._alerts = self._alerts[-1000:]
        
        # 发送通知
        tasks = []
        
        if self._email_config and level in [AlertLevel.ERROR, AlertLevel.CRITICAL]:
            tasks.append(self._send_email(alert))
        
        for webhook_url in self._webhook_urls:
            tasks.append(self._send_webhook(webhook_url, alert))
        
        for handler in self._handlers:
            try:
                if callable(handler):
                    tasks.append(handler(alert))
            except Exception as e:
                logger.error(f"自定义处理器执行失败: {e}")
        
        # 异步发送所有通知
        import asyncio
        await asyncio.gather(*tasks, return_exceptions=True)
        
        logger.info(f"告警已发送: [{level.value}] {title}")
    
    async def _send_email(self, alert: Alert):
        """发送邮件"""
        if not self._email_config:
            return
        
        try:
            msg = MIMEMultipart()
            msg['From'] = self._email_config['from_email']
            msg['To'] = ', '.join(self._email_config['to_emails'])
            msg['Subject'] = f"[{alert.level.value.upper()}] {alert.title}"
            
            body = f"""
告警级别: {alert.level.value}
标题: {alert.title}
消息: {alert.message}
时间: {alert.timestamp.isoformat()}
"""
            if alert.metadata:
                body += f"\n详细信息:\n"
                for key, value in alert.metadata.items():
                    body += f"  {key}: {value}\n"
            
            msg.attach(MIMEText(body, 'plain', 'utf-8'))
            
            # 发送邮件（同步操作，在实际应用中应该使用异步SMTP）
            with smtplib.SMTP(self._email_config['smtp_host'], self._email_config['smtp_port']) as server:
                server.starttls()
                server.login(self._email_config['username'], self._email_config['password'])
                server.send_message(msg)
            
            logger.info("邮件告警发送成功")
            
        except Exception as e:
            logger.error(f"发送邮件告警失败: {e}")
    
    async def _send_webhook(self, url: str, alert: Alert):
        """发送Webhook"""
        try:
            payload = {
                'level': alert.level.value,
                'title': alert.title,
                'message': alert.message,
                'timestamp': alert.timestamp.isoformat(),
                'metadata': alert.metadata or {},
            }
            
            async with aiohttp.ClientSession() as session:
                async with session.post(url, json=payload, timeout=aiohttp.ClientTimeout(total=5)) as response:
                    if response.status == 200:
                        logger.info(f"Webhook告警发送成功: {url}")
                    else:
                        logger.warning(f"Webhook返回非200状态: {response.status}")
                        
        except Exception as e:
            logger.error(f"发送Webhook告警失败 {url}: {e}")
    
    def get_recent_alerts(self, limit: int = 100, level: Optional[AlertLevel] = None) -> List[Dict]:
        """获取最近的告警"""
        alerts = self._alerts[-limit:] if limit > 0 else self._alerts
        
        if level:
            alerts = [a for a in alerts if a.level == level]
        
        return [
            {
                'level': alert.level.value,
                'title': alert.title,
                'message': alert.message,
                'timestamp': alert.timestamp.isoformat(),
                'metadata': alert.metadata,
            }
            for alert in reversed(alerts)
        ]
    
    def enable(self, enabled: bool = True):
        """启用/禁用告警"""
        self._enabled = enabled
        logger.info(f"告警系统{'启用' if enabled else '禁用'}")

