"""
WebSocket广播 - 用于管理面板实时更新
"""
from typing import Set
from fastapi import WebSocket
from loguru import logger
import json


class WebSocketBroadcaster:
    """WebSocket广播器"""
    
    def __init__(self):
        self._connections: Set[WebSocket] = set()
    
    async def connect(self, websocket: WebSocket):
        """连接WebSocket"""
        await websocket.accept()
        self._connections.add(websocket)
        logger.info(f"WebSocket连接，当前连接数: {len(self._connections)}")
    
    async def disconnect(self, websocket: WebSocket):
        """断开WebSocket"""
        self._connections.discard(websocket)
        logger.info(f"WebSocket断开，当前连接数: {len(self._connections)}")
    
    async def broadcast(self, message: dict):
        """广播消息"""
        if not self._connections:
            return
        
        message_json = json.dumps(message)
        disconnected = set()
        
        for connection in self._connections:
            try:
                await connection.send_text(message_json)
            except Exception as e:
                logger.debug(f"广播消息失败: {e}")
                disconnected.add(connection)
        
        # 清理断开的连接
        for connection in disconnected:
            self._connections.discard(connection)
    
    async def send_to(self, websocket: WebSocket, message: dict):
        """发送消息给指定连接"""
        try:
            await websocket.send_text(json.dumps(message))
        except Exception as e:
            logger.error(f"发送消息失败: {e}")
    
    def get_connection_count(self) -> int:
        """获取连接数"""
        return len(self._connections)

