"""
API路由
"""
from fastapi import APIRouter, Depends, HTTPException, WebSocket, WebSocketDisconnect
from typing import List, Optional
from pydantic import BaseModel

from common.models.client import Client, ClientStatus
from common.models.tunnel import Tunnel, TunnelType, TunnelStatus
from common.models.session import Session
from common.models.domain import Domain, DomainStatus
from ..core.gateway.connection_manager import ConnectionManager
from ..core.auth.auth_manager import AuthManager
from ..core.tunnel.tunnel_manager import TunnelManager
from ..core.gateway.websocket_handler import WebSocketHandler
from ..core.domain.domain_manager import DomainManager
from ..core.domain.ssl_manager import SSLManager
from ..core.stats.traffic_stats import TrafficStats
from ..core.monitor.monitor import SystemMonitor
from ..core.quota.quota_manager import QuotaManager, QuotaType, QuotaPeriod
from ..core.cache.cache_manager import CacheManager
from ..core.logging.log_analyzer import LogAnalyzer
from ..core.service_discovery.discovery import ServiceDiscovery
from ..core.security.ip_filter import IPFilter
from ..core.security.rate_limiter import RateLimiter
from ..core.security.anomaly_detector import AnomalyDetector
from ..core.monitoring.prometheus_metrics import PrometheusMetrics
from ..core.alerting.alert_manager import AlertManager, AlertLevel
from ..core.backup.backup_manager import BackupManager
from ..core.cluster.node_manager import NodeManager
from ..core.cluster.session_sync import SessionSync
from ..core.api_auth.api_auth import APIAuth, Permission, Role
from ..core.performance.profiler import PerformanceProfiler, RequestProfiler
from .websocket_broadcast import WebSocketBroadcaster
from fastapi import Request, Response
from fastapi.responses import Response as FastAPIResponse
import time


router = APIRouter()


# 注意：FastAPI中间件应该使用app.middleware，这里先注释，在main.py中注册
# @router.middleware("http")
# async def security_middleware(request: Request, call_next):
    """安全中间件"""
    from ..main import app_state
    
    # IP过滤
    client_ip = request.client.host if request.client else "unknown"
    if app_state.ip_filter and not app_state.ip_filter.is_allowed(client_ip):
        return Response(
            content="IP address not allowed",
            status_code=403
        )
    
    # 速率限制
    if app_state.rate_limiter:
        rate_limit_key = f"{client_ip}:{request.url.path}"
        allowed, limit_info = app_state.rate_limiter.is_allowed(rate_limit_key)
        
        if not allowed:
            return Response(
                content="Rate limit exceeded",
                status_code=429,
                headers={
                    "X-RateLimit-Limit": str(limit_info.get('limit', 0)),
                    "X-RateLimit-Remaining": "0",
                    "Retry-After": str(int(limit_info.get('reset_at', 0) - time.time())),
                }
            )
    
    # 记录请求开始时间
    start_time = time.time()
    
    # 执行请求
    response = await call_next(request)
    
    # 记录请求指标
    if app_state.prometheus_metrics:
        duration = time.time() - start_time
        app_state.prometheus_metrics.record_http_request(
            method=request.method,
            endpoint=str(request.url.path),
            status_code=response.status_code,
            duration=duration
        )
    
    return response


# 依赖注入
def get_connection_manager() -> ConnectionManager:
    """获取连接管理器"""
    from ..main import app_state
    return app_state.connection_manager


def get_auth_manager() -> AuthManager:
    """获取认证管理器"""
    from ..main import app_state
    return app_state.auth_manager


def get_tunnel_manager() -> TunnelManager:
    """获取隧道管理器"""
    from ..main import app_state
    return app_state.tunnel_manager


def get_domain_manager() -> DomainManager:
    """获取域名管理器"""
    from ..main import app_state
    return app_state.domain_manager


def get_ssl_manager() -> SSLManager:
    """获取SSL管理器"""
    from ..main import app_state
    return app_state.ssl_manager


def get_traffic_stats() -> TrafficStats:
    """获取流量统计"""
    from ..main import app_state
    return app_state.traffic_stats


def get_system_monitor() -> SystemMonitor:
    """获取系统监控"""
    from ..main import app_state
    return app_state.system_monitor


def get_quota_manager() -> QuotaManager:
    """获取配额管理器"""
    from ..main import app_state
    return app_state.quota_manager


def get_cache_manager() -> CacheManager:
    """获取缓存管理器"""
    from ..main import app_state
    return app_state.cache_manager


def get_log_analyzer() -> LogAnalyzer:
    """获取日志分析器"""
    from ..main import app_state
    return app_state.log_analyzer


def get_service_discovery() -> ServiceDiscovery:
    """获取服务发现"""
    from ..main import app_state
    return app_state.service_discovery


def get_broadcaster() -> WebSocketBroadcaster:
    """获取WebSocket广播器"""
    from ..main import app_state
    if not hasattr(app_state, 'broadcaster'):
        app_state.broadcaster = WebSocketBroadcaster()
    return app_state.broadcaster


def get_ip_filter() -> IPFilter:
    """获取IP过滤器"""
    from ..main import app_state
    return app_state.ip_filter


def get_rate_limiter() -> RateLimiter:
    """获取速率限制器"""
    from ..main import app_state
    return app_state.rate_limiter


def get_anomaly_detector() -> AnomalyDetector:
    """获取异常检测器"""
    from ..main import app_state
    return app_state.anomaly_detector


def get_prometheus_metrics() -> PrometheusMetrics:
    """获取Prometheus指标"""
    from ..main import app_state
    return app_state.prometheus_metrics


def get_alert_manager() -> AlertManager:
    """获取告警管理器"""
    from ..main import app_state
    return app_state.alert_manager


def get_backup_manager() -> BackupManager:
    """获取备份管理器"""
    from ..main import app_state
    return app_state.backup_manager


def get_node_manager() -> NodeManager:
    """获取节点管理器"""
    from ..main import app_state
    return app_state.node_manager


def get_api_auth() -> APIAuth:
    """获取API认证管理器"""
    from ..main import app_state
    return app_state.api_auth


def get_request_profiler() -> RequestProfiler:
    """获取请求性能分析器"""
    from ..main import app_state
    return app_state.request_profiler


# 请求模型
class CreateTunnelRequest(BaseModel):
    name: str
    tunnel_type: str
    local_host: str
    local_port: int
    remote_port: Optional[int] = None
    domain: Optional[str] = None
    description: Optional[str] = None


class CreateClientRequest(BaseModel):
    name: str
    description: Optional[str] = None


# API路由
@router.get("/")
async def root():
    """根路径"""
    return {"message": "NPS Server API", "version": "1.0.0"}


@router.get("/health")
async def health_check():
    """健康检查"""
    return {"status": "healthy"}


@router.get("/clients", response_model=List[dict])
async def list_clients(
    connection_manager: ConnectionManager = Depends(get_connection_manager),
):
    """获取客户端列表"""
    connections = connection_manager.get_all_connections()
    clients = []
    for client_id, connection in connections.items():
        if connection.client:
            clients.append(connection.client.to_dict())
    return clients


@router.post("/clients", response_model=dict)
async def create_client(
    request: CreateClientRequest,
    auth_manager: AuthManager = Depends(get_auth_manager),
):
    """创建客户端"""
    import uuid
    client_id = str(uuid.uuid4())
    token = auth_manager.create_client_token(client_id, request.name)
    
    return {
        "id": client_id,
        "name": request.name,
        "token": token,
        "description": request.description,
    }


@router.get("/clients/{client_id}", response_model=dict)
async def get_client(
    client_id: str,
    connection_manager: ConnectionManager = Depends(get_connection_manager),
):
    """获取客户端信息"""
    connection = connection_manager.get_connection(client_id)
    if not connection or not connection.client:
        raise HTTPException(status_code=404, detail="客户端不存在")
    return connection.client.to_dict()


@router.get("/clients/{client_id}/tunnels", response_model=List[dict])
async def list_client_tunnels(
    client_id: str,
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """获取客户端的隧道列表"""
    tunnels = tunnel_manager.get_client_tunnels(client_id)
    return [tunnel.to_dict() for tunnel in tunnels]


@router.post("/clients/{client_id}/tunnels", response_model=dict)
async def create_tunnel(
    client_id: str,
    request: CreateTunnelRequest,
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """创建隧道"""
    try:
        tunnel_type = TunnelType(request.tunnel_type)
    except ValueError:
        raise HTTPException(status_code=400, detail=f"不支持的隧道类型: {request.tunnel_type}")
    
    tunnel = tunnel_manager.create_tunnel(
        client_id=client_id,
        name=request.name,
        tunnel_type=tunnel_type,
        local_host=request.local_host,
        local_port=request.local_port,
        remote_port=request.remote_port,
        domain=request.domain,
    )
    
    if request.description:
        tunnel.description = request.description
    
    return tunnel.to_dict()


@router.delete("/tunnels/{tunnel_id}")
async def delete_tunnel(
    tunnel_id: str,
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """删除隧道"""
    tunnel = tunnel_manager.get_tunnel(tunnel_id)
    if not tunnel:
        raise HTTPException(status_code=404, detail="隧道不存在")
    
    tunnel_manager.remove_tunnel(tunnel_id)
    return {"message": "隧道已删除"}


@router.post("/tunnels/{tunnel_id}/activate")
async def activate_tunnel(
    tunnel_id: str,
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """激活隧道"""
    tunnel = tunnel_manager.get_tunnel(tunnel_id)
    if not tunnel:
        raise HTTPException(status_code=404, detail="隧道不存在")
    
    await tunnel_manager.activate_tunnel(tunnel_id)
    return {"message": "隧道已激活"}


@router.post("/tunnels/{tunnel_id}/deactivate")
async def deactivate_tunnel(
    tunnel_id: str,
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """停用隧道"""
    tunnel = tunnel_manager.get_tunnel(tunnel_id)
    if not tunnel:
        raise HTTPException(status_code=404, detail="隧道不存在")
    
    await tunnel_manager.deactivate_tunnel(tunnel_id)
    return {"message": "隧道已停用"}


@router.get("/stats")
async def get_stats(
    connection_manager: ConnectionManager = Depends(get_connection_manager),
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
    traffic_stats: TrafficStats = Depends(get_traffic_stats),
):
    """获取统计信息"""
    return {
        "connections": connection_manager.get_connection_count(),
        "tunnels": len(tunnel_manager._tunnels),
        "active_tunnels": len([t for t in tunnel_manager._tunnels.values() if t.status == TunnelStatus.ACTIVE]),
        "traffic": traffic_stats.get_all_stats(),
    }


@router.get("/clients/{client_id}/domains", response_model=List[dict])
async def list_client_domains(
    client_id: str,
    domain_manager: DomainManager = Depends(get_domain_manager),
):
    """获取客户端的域名列表"""
    domains = domain_manager.get_client_domains(client_id)
    return [domain.to_dict() for domain in domains]


@router.post("/clients/{client_id}/domains", response_model=dict)
async def create_domain(
    client_id: str,
    domain: str,
    tunnel_id: Optional[str] = None,
    ssl_enabled: bool = False,
    domain_manager: DomainManager = Depends(get_domain_manager),
    ssl_manager: SSLManager = Depends(get_ssl_manager),
):
    """创建域名"""
    domain_obj = domain_manager.create_domain(
        client_id=client_id,
        domain=domain,
        tunnel_id=tunnel_id,
        ssl_enabled=ssl_enabled,
    )
    
    # 如果启用SSL，申请证书
    if ssl_enabled:
        await ssl_manager.request_certificate(domain_obj)
    
    return domain_obj.to_dict()


@router.post("/domains/{domain_id}/verify")
async def verify_domain(
    domain_id: str,
    domain_manager: DomainManager = Depends(get_domain_manager),
):
    """验证域名"""
    success = domain_manager.verify_domain(domain_id)
    domain = domain_manager.get_domain(domain_id)
    return {
        "success": success,
        "domain": domain.to_dict() if domain else None,
    }


@router.post("/domains/{domain_id}/ssl/request")
async def request_ssl_certificate(
    domain_id: str,
    domain_manager: DomainManager = Depends(get_domain_manager),
    ssl_manager: SSLManager = Depends(get_ssl_manager),
):
    """申请SSL证书"""
    domain = domain_manager.get_domain(domain_id)
    if not domain:
        raise HTTPException(status_code=404, detail="域名不存在")
    
    success = await ssl_manager.request_certificate(domain)
    return {
        "success": success,
        "domain": domain.to_dict(),
    }


@router.get("/domains/{domain_id}/ssl/info")
async def get_ssl_info(
    domain_id: str,
    domain_manager: DomainManager = Depends(get_domain_manager),
    ssl_manager: SSLManager = Depends(get_ssl_manager),
):
    """获取SSL证书信息"""
    domain = domain_manager.get_domain(domain_id)
    if not domain:
        raise HTTPException(status_code=404, detail="域名不存在")
    
    info = ssl_manager.get_certificate_info(domain)
    return info or {}


@router.get("/clients/{client_id}/traffic")
async def get_client_traffic(
    client_id: str,
    hours: int = 24,
    traffic_stats: TrafficStats = Depends(get_traffic_stats),
):
    """获取客户端流量统计"""
    stats = traffic_stats.get_client_stats(client_id, hours)
    if not stats:
        raise HTTPException(status_code=404, detail="客户端不存在或无流量数据")
    return stats


@router.get("/tunnels/{tunnel_id}/traffic")
async def get_tunnel_traffic(
    tunnel_id: str,
    hours: int = 24,
    traffic_stats: TrafficStats = Depends(get_traffic_stats),
):
    """获取隧道流量统计"""
    stats = traffic_stats.get_tunnel_stats(tunnel_id, hours)
    if not stats:
        raise HTTPException(status_code=404, detail="隧道不存在或无流量数据")
    return stats


@router.get("/monitor/health")
async def get_system_health(
    system_monitor: SystemMonitor = Depends(get_system_monitor),
):
    """获取系统健康状态"""
    health = await system_monitor.check_system_health()
    return health


@router.post("/clients/{client_id}/quotas")
async def set_client_quota(
    client_id: str,
    quota_type: str,
    limit: int,
    period: str = "monthly",
    quota_manager: QuotaManager = Depends(get_quota_manager),
):
    """设置客户端配额"""
    try:
        quota_type_enum = QuotaType(quota_type)
        period_enum = QuotaPeriod(period)
    except ValueError:
        raise HTTPException(status_code=400, detail="无效的配额类型或周期")
    
    quota_manager.set_quota(client_id, quota_type_enum, limit, period_enum)
    return {"message": "配额设置成功"}


@router.get("/clients/{client_id}/quotas")
async def get_client_quotas(
    client_id: str,
    quota_manager: QuotaManager = Depends(get_quota_manager),
):
    """获取客户端配额"""
    quotas = quota_manager.get_client_quotas(client_id)
    return {
        quota_type.value: {
            'limit': quota.limit,
            'used': quota.used,
            'remaining': quota.get_remaining(),
            'usage_percent': quota.get_usage_percent(),
            'period': quota.period.value,
            'reset_at': quota.reset_at.isoformat() if quota.reset_at else None,
        }
        for quota_type, quota in quotas.items()
    }


@router.get("/logs/files")
async def list_log_files(
    log_analyzer: LogAnalyzer = Depends(get_log_analyzer),
):
    """获取日志文件列表"""
    files = log_analyzer.get_log_files()
    return {"files": files}


@router.get("/logs/analyze/{log_file}")
async def analyze_logs(
    log_file: str,
    log_analyzer: LogAnalyzer = Depends(get_log_analyzer),
):
    """分析日志文件"""
    stats = log_analyzer.analyze_logs(log_file)
    return stats


@router.get("/logs/search/{log_file}")
async def search_logs(
    log_file: str,
    keyword: str,
    limit: int = 100,
    log_analyzer: LogAnalyzer = Depends(get_log_analyzer),
):
    """搜索日志"""
    results = log_analyzer.search_logs(log_file, keyword, limit)
    return {"results": results}


@router.get("/logs/errors/{log_file}")
async def get_recent_errors(
    log_file: str,
    hours: int = 24,
    limit: int = 50,
    log_analyzer: LogAnalyzer = Depends(get_log_analyzer),
):
    """获取最近的错误日志"""
    errors = log_analyzer.get_recent_errors(log_file, hours, limit)
    return {"errors": errors}


@router.post("/discovery/scan")
async def scan_services(
    host: str = "127.0.0.1",
    ports: Optional[List[int]] = None,
    service_discovery: ServiceDiscovery = Depends(get_service_discovery),
):
    """扫描本地服务"""
    services = await service_discovery.scan_local_services(host, ports)
    return {
        "services": [
            {
                'name': s.name,
                'host': s.host,
                'port': s.port,
                'protocol': s.protocol,
                'status': s.status,
            }
            for s in services
        ]
    }


@router.get("/cache/stats")
async def get_cache_stats(
    cache_manager: CacheManager = Depends(get_cache_manager),
):
    """获取缓存统计"""
    return cache_manager.get_stats()


@router.post("/security/whitelist")
async def add_whitelist(
    ip_or_cidr: str,
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """添加IP白名单"""
    ip_filter.add_whitelist(ip_or_cidr)
    return {"message": "白名单已添加"}


@router.delete("/security/whitelist/{ip_or_cidr}")
async def remove_whitelist(
    ip_or_cidr: str,
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """移除IP白名单"""
    ip_filter.remove_whitelist(ip_or_cidr)
    return {"message": "白名单已移除"}


@router.get("/security/whitelist")
async def get_whitelist(
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """获取IP白名单"""
    return {"whitelist": ip_filter.get_whitelist()}


@router.post("/security/blacklist")
async def add_blacklist(
    ip_or_cidr: str,
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """添加IP黑名单"""
    ip_filter.add_blacklist(ip_or_cidr)
    return {"message": "黑名单已添加"}


@router.delete("/security/blacklist/{ip_or_cidr}")
async def remove_blacklist(
    ip_or_cidr: str,
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """移除IP黑名单"""
    ip_filter.remove_blacklist(ip_or_cidr)
    return {"message": "黑名单已移除"}


@router.get("/security/blacklist")
async def get_blacklist(
    ip_filter: IPFilter = Depends(get_ip_filter),
):
    """获取IP黑名单"""
    return {"blacklist": ip_filter.get_blacklist()}


@router.post("/security/rate-limit")
async def set_rate_limit(
    key: str,
    max_requests: int,
    window_seconds: int,
    rate_limiter: RateLimiter = Depends(get_rate_limiter),
):
    """设置速率限制"""
    rate_limiter.set_limit(key, max_requests, window_seconds)
    return {"message": "速率限制已设置"}


@router.get("/security/anomalies")
async def get_anomalies(
    minutes: int = 60,
    severity: Optional[str] = None,
    anomaly_detector: AnomalyDetector = Depends(get_anomaly_detector),
):
    """获取异常事件"""
    anomalies = anomaly_detector.get_recent_anomalies(minutes, severity)
    return {"anomalies": anomalies}


@router.get("/metrics")
async def get_prometheus_metrics(
    metrics: PrometheusMetrics = Depends(get_prometheus_metrics),
):
    """获取Prometheus指标"""
    return FastAPIResponse(
        content=metrics.get_metrics(),
        media_type="text/plain"
    )


@router.get("/alerts")
async def get_alerts(
    limit: int = 100,
    level: Optional[str] = None,
    alert_manager: AlertManager = Depends(get_alert_manager),
):
    """获取告警列表"""
    from ..core.alerting.alert_manager import AlertLevel
    level_enum = AlertLevel(level) if level else None
    alerts = alert_manager.get_recent_alerts(limit, level_enum)
    return {"alerts": alerts}


@router.post("/backups")
async def create_backup(
    name: Optional[str] = None,
    backup_manager: BackupManager = Depends(get_backup_manager),
):
    """创建备份"""
    backup_path = backup_manager.create_backup(name)
    return {"message": "备份创建成功", "path": backup_path}


@router.get("/backups")
async def list_backups(
    backup_manager: BackupManager = Depends(get_backup_manager),
):
    """列出所有备份"""
    backups = backup_manager.list_backups()
    return {"backups": backups}


@router.post("/backups/{backup_name}/restore")
async def restore_backup(
    backup_name: str,
    backup_manager: BackupManager = Depends(get_backup_manager),
):
    """恢复备份"""
    success = backup_manager.restore_backup(backup_name)
    if success:
        return {"message": "备份恢复成功"}
    else:
        raise HTTPException(status_code=400, detail="备份恢复失败")


@router.delete("/backups/{backup_name}")
async def delete_backup(
    backup_name: str,
    backup_manager: BackupManager = Depends(get_backup_manager),
):
    """删除备份"""
    success = backup_manager.delete_backup(backup_name)
    if success:
        return {"message": "备份已删除"}
    else:
        raise HTTPException(status_code=404, detail="备份不存在")


@router.post("/cluster/heartbeat")
async def receive_heartbeat(
    node_info: dict,
    node_manager: NodeManager = Depends(get_node_manager),
):
    """接收节点心跳"""
    await node_manager.receive_heartbeat(node_info)
    return {"status": "ok"}


@router.get("/cluster/nodes")
async def get_cluster_nodes(
    node_manager: NodeManager = Depends(get_node_manager),
):
    """获取集群节点信息"""
    return node_manager.get_cluster_stats()


@router.post("/cluster/nodes")
async def add_cluster_node(
    node_id: str,
    address: str,
    port: int,
    role: str = "worker",
    node_manager: NodeManager = Depends(get_node_manager),
):
    """添加集群节点"""
    node_manager.add_node(node_id, address, port, role)
    return {"message": "节点已添加"}


@router.post("/api/auth/token")
async def create_api_token(
    username: str,
    role: str = "viewer",
    api_auth: APIAuth = Depends(get_api_auth),
):
    """创建API Token"""
    try:
        role_enum = Role(role)
    except ValueError:
        raise HTTPException(status_code=400, detail=f"无效的角色: {role}")
    
    token = api_auth.create_api_token(username, role_enum)
    return {
        "token": token,
        "username": username,
        "role": role,
        "expires_in": 86400,  # 24小时
    }


@router.get("/performance/stats")
async def get_performance_stats(
    request_profiler: RequestProfiler = Depends(get_request_profiler),
):
    """获取性能统计"""
    return {
        "endpoints": request_profiler.get_all_stats()
    }


@router.get("/performance/slowest")
async def get_slowest_functions(
    limit: int = 10,
):
    """获取最慢的函数"""
    from ..main import app_state
    if app_state.performance_profiler:
        slowest = app_state.performance_profiler.get_slowest_functions(limit)
        return {
            "functions": [
                {"name": name, "duration": duration}
                for name, duration in slowest
            ]
        }
    return {"functions": []}


@router.websocket("/ws")
async def websocket_endpoint(
    websocket: WebSocket,
    connection_manager: ConnectionManager = Depends(get_connection_manager),
    auth_manager: AuthManager = Depends(get_auth_manager),
    tunnel_manager: TunnelManager = Depends(get_tunnel_manager),
):
    """WebSocket端点（客户端连接）"""
    client_host = websocket.client.host if websocket.client else "unknown"
    
    handler = WebSocketHandler(
        connection_manager=connection_manager,
        auth_manager=auth_manager,
        tunnel_manager=tunnel_manager,
    )
    
    await handler.handle_connection(websocket, client_host)


@router.websocket("/ws/admin")
async def admin_websocket_endpoint(
    websocket: WebSocket,
    broadcaster: WebSocketBroadcaster = Depends(get_broadcaster),
):
    """WebSocket端点（管理面板）"""
    await broadcaster.connect(websocket)
    
    # 发送初始数据
    try:
        from ..main import app_state
        
        # 发送统计信息
        stats = {
            'type': 'stats_update',
            'connections': app_state.connection_manager.get_connection_count() if app_state.connection_manager else 0,
            'tunnels': len(app_state.tunnel_manager._tunnels) if app_state.tunnel_manager else 0,
        }
        await broadcaster.send_to(websocket, stats)
        
        # 定期发送更新
        import asyncio
        while True:
            await asyncio.sleep(30)  # 每30秒更新一次
            
            # 发送统计更新
            if app_state.connection_manager and app_state.tunnel_manager:
                stats = {
                    'type': 'stats_update',
                    'connections': app_state.connection_manager.get_connection_count(),
                    'tunnels': len(app_state.tunnel_manager._tunnels),
                    'timestamp': time.time(),
                }
                await broadcaster.send_to(websocket, stats)
            
    except WebSocketDisconnect:
        pass
        logger.debug("管理面板WebSocket断开")
    except Exception as e:
        logger.error(f"管理面板WebSocket错误: {e}")
    finally:
        await broadcaster.disconnect(websocket)

