#!/bin/bash

# NPS系统状态查看脚本
# 使用方法: ./scripts/status.sh

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 获取脚本所在目录
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"

echo -e "${BLUE}"
echo "=========================================="
echo "    NPS 系统状态"
echo "=========================================="
echo -e "${NC}"

# 检查服务器状态
echo -e "${YELLOW}服务器状态:${NC}"
SERVER_PID_FILE="$PROJECT_ROOT/logs/server.pid"
if [ -f "$SERVER_PID_FILE" ]; then
    SERVER_PID=$(cat "$SERVER_PID_FILE")
    if ps -p $SERVER_PID > /dev/null 2>&1; then
        echo -e "${GREEN}✓ 服务器运行中（PID: $SERVER_PID）${NC}"
        
        # 检查端口
        PORT=${PORT:-"8080"}
        if lsof -Pi :$PORT -sTCP:LISTEN -t >/dev/null 2>&1; then
            echo -e "${GREEN}✓ 端口 $PORT 正在监听${NC}"
        else
            echo -e "${YELLOW}⚠ 端口 $PORT 未监听${NC}"
        fi
    else
        echo -e "${RED}✗ 服务器未运行（PID文件存在但进程不存在）${NC}"
    fi
else
    # 尝试通过进程名查找
    SERVER_PIDS=$(pgrep -f "start_server.py" || true)
    if [ -n "$SERVER_PIDS" ]; then
        echo -e "${GREEN}✓ 服务器运行中（PID: $SERVER_PIDS）${NC}"
    else
        echo -e "${RED}✗ 服务器未运行${NC}"
    fi
fi

echo ""

# 检查客户端状态
echo -e "${YELLOW}客户端状态:${NC}"
CLIENT_PID_FILE="$PROJECT_ROOT/logs/client.pid"
if [ -f "$CLIENT_PID_FILE" ]; then
    CLIENT_PID=$(cat "$CLIENT_PID_FILE")
    if ps -p $CLIENT_PID > /dev/null 2>&1; then
        echo -e "${GREEN}✓ 客户端运行中（PID: $CLIENT_PID）${NC}"
    else
        echo -e "${RED}✗ 客户端未运行（PID文件存在但进程不存在）${NC}"
    fi
else
    # 尝试通过进程名查找
    CLIENT_PIDS=$(pgrep -f "start_client.py\|start_client_gui.py" || true)
    if [ -n "$CLIENT_PIDS" ]; then
        echo -e "${GREEN}✓ 客户端运行中（PID: $CLIENT_PIDS）${NC}"
    else
        echo -e "${RED}✗ 客户端未运行${NC}"
    fi
fi

echo ""

# 检查数据库连接
echo -e "${YELLOW}数据库状态:${NC}"
if command -v psql &> /dev/null; then
    DATABASE_URL=${DATABASE_URL:-"postgresql://nps:nps@localhost:5432/nps"}
    # 提取连接信息
    if [[ $DATABASE_URL == postgresql://* ]]; then
        DB_INFO=$(echo $DATABASE_URL | sed 's|postgresql://||' | sed 's|/.*||')
        DB_USER=$(echo $DB_INFO | cut -d: -f1)
        DB_PASS=$(echo $DB_INFO | cut -d: -f2 | cut -d@ -f1)
        DB_HOST=$(echo $DB_INFO | cut -d@ -f2 | cut -d: -f1)
        DB_PORT=$(echo $DB_INFO | cut -d: -f2 | cut -d: -f2)
        
        if PGPASSWORD=$DB_PASS psql -h $DB_HOST -p ${DB_PORT:-5432} -U $DB_USER -d nps -c "SELECT 1" > /dev/null 2>&1; then
            echo -e "${GREEN}✓ 数据库连接正常${NC}"
        else
            echo -e "${RED}✗ 数据库连接失败${NC}"
        fi
    fi
else
    echo -e "${YELLOW}⚠ psql未安装，跳过数据库检查${NC}"
fi

echo ""

# 检查Redis连接
echo -e "${YELLOW}Redis状态:${NC}"
REDIS_URL=${REDIS_URL:-"redis://localhost:6379/0"}
if command -v redis-cli &> /dev/null; then
    REDIS_HOST=$(echo $REDIS_URL | sed 's|redis://||' | cut -d: -f1)
    REDIS_PORT=$(echo $REDIS_URL | sed 's|redis://||' | cut -d: -f2 | cut -d/ -f1)
    
    if redis-cli -h ${REDIS_HOST:-localhost} -p ${REDIS_PORT:-6379} ping > /dev/null 2>&1; then
        echo -e "${GREEN}✓ Redis连接正常${NC}"
    else
        echo -e "${RED}✗ Redis连接失败${NC}"
    fi
else
    echo -e "${YELLOW}⚠ redis-cli未安装，跳过Redis检查${NC}"
fi

echo ""
echo -e "${BLUE}=========================================="
echo -e "${NC}"

