#!/bin/bash

# NPS服务器一键启动脚本
# 使用方法: ./scripts/start_server.sh [选项]
# 选项:
#   -d, --daemon    后台运行
#   -h, --help      显示帮助信息

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 获取脚本所在目录
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
cd "$PROJECT_ROOT"

# 默认配置
DAEMON=false
PID_FILE="$PROJECT_ROOT/logs/server.pid"
LOG_FILE="$PROJECT_ROOT/logs/server.log"

# 解析命令行参数
while [[ $# -gt 0 ]]; do
    case $1 in
        -d|--daemon)
            DAEMON=true
            shift
            ;;
        -h|--help)
            echo "NPS服务器启动脚本"
            echo ""
            echo "使用方法: $0 [选项]"
            echo ""
            echo "选项:"
            echo "  -d, --daemon    后台运行"
            echo "  -h, --help      显示帮助信息"
            exit 0
            ;;
        *)
            echo -e "${RED}未知参数: $1${NC}"
            exit 1
            ;;
    esac
done

# 打印横幅
echo -e "${BLUE}"
echo "=========================================="
echo "    NPS 服务器启动脚本"
echo "=========================================="
echo -e "${NC}"

# 检查Python环境
echo -e "${YELLOW}[1/6] 检查Python环境...${NC}"
if ! command -v python3 &> /dev/null; then
    echo -e "${RED}错误: 未找到Python3，请先安装Python3${NC}"
    exit 1
fi

PYTHON_VERSION=$(python3 --version | cut -d' ' -f2)
echo -e "${GREEN}✓ Python版本: $PYTHON_VERSION${NC}"

# 检查虚拟环境
echo -e "${YELLOW}[2/6] 检查虚拟环境...${NC}"
if [ -d "venv" ]; then
    echo -e "${GREEN}✓ 发现虚拟环境，激活中...${NC}"
    source venv/bin/activate
elif [ -d ".venv" ]; then
    echo -e "${GREEN}✓ 发现虚拟环境，激活中...${NC}"
    source .venv/bin/activate
else
    echo -e "${YELLOW}⚠ 未找到虚拟环境，使用系统Python${NC}"
fi

# 检查依赖
echo -e "${YELLOW}[3/6] 检查依赖...${NC}"
if ! python3 -c "import fastapi" 2>/dev/null; then
    echo -e "${YELLOW}⚠ 依赖未安装，正在安装...${NC}"
    pip install -r requirements.txt -q
fi
echo -e "${GREEN}✓ 依赖检查完成${NC}"

# 创建必要目录
echo -e "${YELLOW}[4/6] 创建必要目录...${NC}"
mkdir -p logs certs backups
echo -e "${GREEN}✓ 目录创建完成${NC}"

# 检查环境变量
echo -e "${YELLOW}[5/6] 检查环境变量...${NC}"
if [ -f ".env" ]; then
    echo -e "${GREEN}✓ 发现.env文件，加载环境变量${NC}"
    export $(cat .env | grep -v '^#' | xargs)
else
    echo -e "${YELLOW}⚠ 未找到.env文件，使用默认配置${NC}"
    export HOST=${HOST:-"0.0.0.0"}
    export PORT=${PORT:-"8080"}
    export SECRET_KEY=${SECRET_KEY:-"change-me-in-production"}
    export DATABASE_URL=${DATABASE_URL:-"postgresql://nps:nps@localhost:5432/nps"}
    export REDIS_URL=${REDIS_URL:-"redis://localhost:6379/0"}
fi
echo -e "${GREEN}✓ 环境变量配置完成${NC}"

# 检查端口是否被占用
echo -e "${YELLOW}[6/6] 检查端口占用...${NC}"
PORT=${PORT:-"8080"}
if lsof -Pi :$PORT -sTCP:LISTEN -t >/dev/null 2>&1 ; then
    echo -e "${RED}错误: 端口 $PORT 已被占用${NC}"
    echo -e "${YELLOW}提示: 可以设置 PORT 环境变量使用其他端口${NC}"
    exit 1
fi
echo -e "${GREEN}✓ 端口 $PORT 可用${NC}"

# 启动服务器
echo ""
echo -e "${BLUE}=========================================="
echo "    启动NPS服务器"
echo "=========================================="
echo -e "${NC}"

if [ "$DAEMON" = true ]; then
    echo -e "${YELLOW}后台模式启动...${NC}"
    nohup python3 start_server.py > "$LOG_FILE" 2>&1 &
    SERVER_PID=$!
    echo $SERVER_PID > "$PID_FILE"
    echo -e "${GREEN}✓ 服务器已启动（PID: $SERVER_PID）${NC}"
    echo -e "${GREEN}✓ 日志文件: $LOG_FILE${NC}"
    echo -e "${GREEN}✓ PID文件: $PID_FILE${NC}"
    echo ""
    echo -e "${BLUE}使用以下命令查看日志:${NC}"
    echo "  tail -f $LOG_FILE"
    echo ""
    echo -e "${BLUE}使用以下命令停止服务器:${NC}"
    echo "  ./scripts/stop_server.sh"
else
    echo -e "${GREEN}前台模式启动...${NC}"
    echo -e "${YELLOW}按 Ctrl+C 停止服务器${NC}"
    echo ""
    python3 start_server.py
fi

