#!/bin/bash

# NPS客户端一键启动脚本（命令行）
# 使用方法: ./scripts/start_client.sh [选项]
# 选项:
#   -d, --daemon    后台运行
#   -h, --help      显示帮助信息

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 获取脚本所在目录
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
cd "$PROJECT_ROOT"

# 默认配置
DAEMON=false
PID_FILE="$PROJECT_ROOT/logs/client.pid"
LOG_FILE="$PROJECT_ROOT/logs/client.log"

# 解析命令行参数
while [[ $# -gt 0 ]]; do
    case $1 in
        -d|--daemon)
            DAEMON=true
            shift
            ;;
        -h|--help)
            echo "NPS客户端启动脚本（命令行）"
            echo ""
            echo "使用方法: $0 [选项]"
            echo ""
            echo "选项:"
            echo "  -d, --daemon    后台运行"
            echo "  -h, --help      显示帮助信息"
            exit 0
            ;;
        *)
            echo -e "${RED}未知参数: $1${NC}"
            exit 1
            ;;
    esac
done

# 打印横幅
echo -e "${BLUE}"
echo "=========================================="
echo "    NPS 客户端启动脚本（命令行）"
echo "=========================================="
echo -e "${NC}"

# 检查Python环境
echo -e "${YELLOW}[1/5] 检查Python环境...${NC}"
if ! command -v python3 &> /dev/null; then
    echo -e "${RED}错误: 未找到Python3，请先安装Python3${NC}"
    exit 1
fi

PYTHON_VERSION=$(python3 --version | cut -d' ' -f2)
echo -e "${GREEN}✓ Python版本: $PYTHON_VERSION${NC}"

# 检查虚拟环境
echo -e "${YELLOW}[2/5] 检查虚拟环境...${NC}"
if [ -d "venv" ]; then
    echo -e "${GREEN}✓ 发现虚拟环境，激活中...${NC}"
    source venv/bin/activate
elif [ -d ".venv" ]; then
    echo -e "${GREEN}✓ 发现虚拟环境，激活中...${NC}"
    source .venv/bin/activate
else
    echo -e "${YELLOW}⚠ 未找到虚拟环境，使用系统Python${NC}"
fi

# 检查依赖
echo -e "${YELLOW}[3/5] 检查依赖...${NC}"
if ! python3 -c "import aiohttp" 2>/dev/null; then
    echo -e "${YELLOW}⚠ 依赖未安装，正在安装...${NC}"
    pip install -r requirements.txt -q
fi
echo -e "${GREEN}✓ 依赖检查完成${NC}"

# 创建必要目录
echo -e "${YELLOW}[4/5] 创建必要目录...${NC}"
mkdir -p logs
echo -e "${GREEN}✓ 目录创建完成${NC}"

# 检查配置文件
echo -e "${YELLOW}[5/5] 检查配置文件...${NC}"
if [ ! -f "client/config.yaml" ] && [ ! -f ".env" ]; then
    echo -e "${YELLOW}⚠ 未找到配置文件，请先配置客户端信息${NC}"
    echo ""
    echo -e "${BLUE}请创建配置文件或设置环境变量:${NC}"
    echo "  - client/config.yaml"
    echo "  - .env"
    echo ""
    echo -e "${BLUE}必需的配置项:${NC}"
    echo "  - client_id: 客户端ID"
    echo "  - token: 客户端Token"
    echo "  - servers: 服务器地址列表"
    exit 1
fi
echo -e "${GREEN}✓ 配置文件检查完成${NC}"

# 启动客户端
echo ""
echo -e "${BLUE}=========================================="
echo "    启动NPS客户端"
echo "=========================================="
echo -e "${NC}"

if [ "$DAEMON" = true ]; then
    echo -e "${YELLOW}后台模式启动...${NC}"
    nohup python3 start_client.py > "$LOG_FILE" 2>&1 &
    CLIENT_PID=$!
    echo $CLIENT_PID > "$PID_FILE"
    echo -e "${GREEN}✓ 客户端已启动（PID: $CLIENT_PID）${NC}"
    echo -e "${GREEN}✓ 日志文件: $LOG_FILE${NC}"
    echo -e "${GREEN}✓ PID文件: $PID_FILE${NC}"
    echo ""
    echo -e "${BLUE}使用以下命令查看日志:${NC}"
    echo "  tail -f $LOG_FILE"
    echo ""
    echo -e "${BLUE}使用以下命令停止客户端:${NC}"
    echo "  ./scripts/stop_client.sh"
else
    echo -e "${GREEN}前台模式启动...${NC}"
    echo -e "${YELLOW}按 Ctrl+C 停止客户端${NC}"
    echo ""
    python3 start_client.py
fi

