#!/bin/bash

# NPS系统一键启动脚本（服务器+客户端）
# 使用方法: ./scripts/start_all.sh [选项]
# 选项:
#   -s, --server-only    仅启动服务器
#   -c, --client-only    仅启动客户端
#   -g, --gui            启动GUI客户端（而非命令行客户端）
#   -d, --daemon         后台运行
#   -h, --help           显示帮助信息

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 获取脚本所在目录
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
cd "$PROJECT_ROOT"

# 默认配置
START_SERVER=true
START_CLIENT=true
USE_GUI=false
DAEMON=false

# 解析命令行参数
while [[ $# -gt 0 ]]; do
    case $1 in
        -s|--server-only)
            START_SERVER=true
            START_CLIENT=false
            shift
            ;;
        -c|--client-only)
            START_SERVER=false
            START_CLIENT=true
            shift
            ;;
        -g|--gui)
            USE_GUI=true
            shift
            ;;
        -d|--daemon)
            DAEMON=true
            shift
            ;;
        -h|--help)
            echo "NPS系统一键启动脚本"
            echo ""
            echo "使用方法: $0 [选项]"
            echo ""
            echo "选项:"
            echo "  -s, --server-only    仅启动服务器"
            echo "  -c, --client-only    仅启动客户端"
            echo "  -g, --gui            启动GUI客户端（而非命令行客户端）"
            echo "  -d, --daemon         后台运行"
            echo "  -h, --help           显示帮助信息"
            exit 0
            ;;
        *)
            echo -e "${RED}未知参数: $1${NC}"
            exit 1
            ;;
    esac
done

# 打印横幅
echo -e "${BLUE}"
echo "=========================================="
echo "    NPS 系统一键启动脚本"
echo "=========================================="
echo -e "${NC}"

# 启动服务器
if [ "$START_SERVER" = true ]; then
    echo ""
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BLUE}  启动服务器${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    
    if [ "$DAEMON" = true ]; then
        bash "$SCRIPT_DIR/start_server.sh" --daemon
    else
        # 后台启动服务器
        bash "$SCRIPT_DIR/start_server.sh" --daemon &
        SERVER_PID=$!
        echo -e "${GREEN}服务器已在后台启动（PID: $SERVER_PID）${NC}"
        sleep 3  # 等待服务器启动
    fi
fi

# 启动客户端
if [ "$START_CLIENT" = true ]; then
    echo ""
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    if [ "$USE_GUI" = true ]; then
        echo -e "${BLUE}  启动客户端（GUI）${NC}"
        echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
        bash "$SCRIPT_DIR/start_client_gui.sh"
    else
        echo -e "${BLUE}  启动客户端（命令行）${NC}"
        echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
        if [ "$DAEMON" = true ]; then
            bash "$SCRIPT_DIR/start_client.sh" --daemon
        else
            bash "$SCRIPT_DIR/start_client.sh"
        fi
    fi
fi

# 如果都是后台模式，显示状态
if [ "$DAEMON" = true ] && [ "$START_SERVER" = true ] && [ "$START_CLIENT" = true ]; then
    echo ""
    echo -e "${GREEN}=========================================="
    echo "    所有服务已启动"
    echo "=========================================="
    echo -e "${NC}"
    echo -e "${BLUE}查看服务器日志:${NC}"
    echo "  tail -f logs/server.log"
    echo ""
    echo -e "${BLUE}查看客户端日志:${NC}"
    echo "  tail -f logs/client.log"
    echo ""
    echo -e "${BLUE}停止所有服务:${NC}"
    echo "  ./scripts/stop_all.sh"
fi

