"""
WebSocket客户端使用示例
"""
import asyncio
import websockets
import json
from typing import Optional


class NPSWebSocketClient:
    """NPS WebSocket客户端示例"""
    
    def __init__(self, ws_url: str, client_id: str, token: str):
        self.ws_url = ws_url
        self.client_id = client_id
        self.token = token
        self.ws: Optional[websockets.WebSocketServerProtocol] = None
    
    async def connect(self):
        """连接WebSocket"""
        self.ws = await websockets.connect(f"{self.ws_url}/ws/client")
        
        # 发送认证
        auth_message = {
            "type": "AUTH_REQUEST",
            "data": {
                "client_id": self.client_id,
                "token": self.token
            }
        }
        await self.ws.send(json.dumps(auth_message))
        
        # 等待认证响应
        response = await self.ws.recv()
        auth_response = json.loads(response)
        
        if auth_response.get("type") == "AUTH_RESPONSE":
            if auth_response.get("data", {}).get("success"):
                print("认证成功")
                return True
            else:
                print(f"认证失败: {auth_response.get('data', {}).get('error')}")
                return False
        
        return False
    
    async def send_tunnel_data(self, tunnel_id: str, data: bytes):
        """发送隧道数据"""
        if not self.ws:
            raise Exception("未连接")
        
        message = {
            "type": "TUNNEL_DATA",
            "data": {
                "tunnel_id": tunnel_id,
                "data": data.hex()  # 转换为十六进制字符串
            }
        }
        await self.ws.send(json.dumps(message))
    
    async def listen(self):
        """监听消息"""
        if not self.ws:
            raise Exception("未连接")
        
        try:
            async for message in self.ws:
                data = json.loads(message)
                await self.handle_message(data)
        except websockets.exceptions.ConnectionClosed:
            print("连接已关闭")
    
    async def handle_message(self, message: Dict):
        """处理消息"""
        msg_type = message.get("type")
        
        if msg_type == "TUNNEL_DATA":
            # 处理隧道数据
            tunnel_id = message.get("data", {}).get("tunnel_id")
            data = bytes.fromhex(message.get("data", {}).get("data", ""))
            print(f"收到隧道数据: {tunnel_id}, 大小: {len(data)}字节")
        
        elif msg_type == "HEARTBEAT":
            # 响应心跳
            await self.ws.send(json.dumps({"type": "HEARTBEAT_RESPONSE"}))
        
        elif msg_type == "CONTROL":
            # 处理控制消息
            print(f"收到控制消息: {message.get('data')}")
    
    async def close(self):
        """关闭连接"""
        if self.ws:
            await self.ws.close()


async def main():
    """示例主函数"""
    client = NPSWebSocketClient(
        ws_url="ws://localhost:8080",
        client_id="your-client-id",
        token="your-token"
    )
    
    # 连接
    if await client.connect():
        # 发送心跳
        await client.send_heartbeat()
        
        # 监听消息（在后台运行）
        listen_task = asyncio.create_task(client.listen())
        
        # 发送一些测试数据
        await asyncio.sleep(1)
        await client.send_tunnel_data("tunnel-1", b"test data")
        
        # 运行一段时间
        await asyncio.sleep(10)
        
        # 关闭连接
        await client.close()
        listen_task.cancel()


if __name__ == "__main__":
    asyncio.run(main())

