"""
API客户端使用示例
"""
import asyncio
import aiohttp
from typing import Dict, List


class NPSAPIClient:
    """NPS API客户端示例"""
    
    def __init__(self, base_url: str = "http://localhost:8080", api_token: str = None):
        self.base_url = base_url
        self.api_token = api_token
        self.headers = {
            "Content-Type": "application/json"
        }
        if api_token:
            self.headers["Authorization"] = f"Bearer {api_token}"
    
    async def create_client(self, name: str) -> Dict:
        """创建客户端"""
        async with aiohttp.ClientSession() as session:
            async with session.post(
                f"{self.base_url}/api/clients",
                json={"name": name},
                headers=self.headers
            ) as resp:
                return await resp.json()
    
    async def get_clients(self) -> List[Dict]:
        """获取客户端列表"""
        async with aiohttp.ClientSession() as session:
            async with session.get(
                f"{self.base_url}/api/clients",
                headers=self.headers
            ) as resp:
                return await resp.json()
    
    async def create_tunnel(self, client_id: str, tunnel_config: Dict) -> Dict:
        """创建隧道"""
        async with aiohttp.ClientSession() as session:
            async with session.post(
                f"{self.base_url}/api/clients/{client_id}/tunnels",
                json=tunnel_config,
                headers=self.headers
            ) as resp:
                return await resp.json()
    
    async def get_traffic_stats(self, client_id: str = None) -> Dict:
        """获取流量统计"""
        url = f"{self.base_url}/api/stats/traffic"
        if client_id:
            url += f"?client_id={client_id}"
        
        async with aiohttp.ClientSession() as session:
            async with session.get(url, headers=self.headers) as resp:
                return await resp.json()
    
    async def get_system_health(self) -> Dict:
        """获取系统健康状态"""
        async with aiohttp.ClientSession() as session:
            async with session.get(
                f"{self.base_url}/api/monitor/health",
                headers=self.headers
            ) as resp:
                return await resp.json()


async def main():
    """示例主函数"""
    # 创建API客户端
    client = NPSAPIClient(
        base_url="http://localhost:8080",
        api_token="your-api-token-here"  # 可选
    )
    
    # 创建客户端
    new_client = await client.create_client("My Client")
    print(f"创建客户端: {new_client}")
    
    client_id = new_client.get("id")
    
    # 获取客户端列表
    clients = await client.get_clients()
    print(f"客户端列表: {clients}")
    
    # 创建隧道
    tunnel = await client.create_tunnel(client_id, {
        "name": "SSH Tunnel",
        "tunnel_type": "tcp",
        "local_host": "127.0.0.1",
        "local_port": 22,
        "remote_port": 2222
    })
    print(f"创建隧道: {tunnel}")
    
    # 获取流量统计
    stats = await client.get_traffic_stats(client_id)
    print(f"流量统计: {stats}")
    
    # 获取系统健康状态
    health = await client.get_system_health()
    print(f"系统健康: {health}")


if __name__ == "__main__":
    asyncio.run(main())

