"""
NPS客户端主程序
"""
import asyncio
import sys
from loguru import logger

from common.utils.logger import setup_logger
from .core.config.config_manager import ConfigManager
from .core.connection.connection_manager import ClientConnectionManager
from .core.proxy.local_proxy import LocalProxy


class NPSCClient:
    """NPS客户端"""
    
    def __init__(self):
        self.config_manager = ConfigManager()
        self.connection_manager: Optional[ClientConnectionManager] = None
        self.local_proxy: Optional[LocalProxy] = None
        self._running = False
    
    async def initialize(self):
        """初始化客户端"""
        server_urls = self.config_manager.get_server_urls()
        client_id = self.config_manager.get_client_id()
        token = self.config_manager.get_client_token()
        
        if not client_id or not token:
            logger.error("客户端ID或Token未配置，请先配置客户端信息")
            return False
        
        if not server_urls:
            logger.error("未配置服务器地址")
            return False
        
        # 创建连接管理器（支持多服务器）
        self.connection_manager = ClientConnectionManager(
            server_urls=server_urls,
            client_id=client_id,
            token=token,
        )
        
        # 设置回调
        self.connection_manager.set_on_connect(self._on_connect)
        self.connection_manager.set_on_disconnect(self._on_disconnect)
        self.connection_manager.set_on_message(self._on_message)
        
        # 创建本地代理
        self.local_proxy = LocalProxy(self.connection_manager)
        
        return True
    
    async def _on_connect(self):
        """连接成功回调"""
        logger.info("已连接到服务器")
    
    async def _on_disconnect(self):
        """断开连接回调"""
        logger.warning("与服务器断开连接")
    
    async def _on_message(self, message):
        """消息回调"""
        logger.debug(f"收到消息: {message.type}")
        # TODO: 处理各种消息类型
    
    async def start(self):
        """启动客户端"""
        if not await self.initialize():
            return
        
        self._running = True
        
        # 启动连接管理器
        try:
            await self.connection_manager.start()
        except KeyboardInterrupt:
            logger.info("收到中断信号，正在关闭...")
        finally:
            await self.stop()
    
    async def stop(self):
        """停止客户端"""
        self._running = False
        if self.connection_manager:
            await self.connection_manager.stop()
        logger.info("客户端已停止")


def main():
    """主函数"""
    # 配置日志
    setup_logger(level="INFO", log_file="logs/client.log")
    
    # 创建客户端
    client = NPSCClient()
    
    # 运行客户端
    try:
        asyncio.run(client.start())
    except KeyboardInterrupt:
        logger.info("程序已退出")


if __name__ == "__main__":
    main()

