"""
主窗口
"""
import sys
import asyncio
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QPushButton, QTextEdit, QTableWidget, QTableWidgetItem,
    QTabWidget, QGroupBox, QLineEdit, QSpinBox, QComboBox, QMessageBox,
    QTreeWidget, QTreeWidgetItem,
)
from PyQt6.QtCore import Qt, QThread, pyqtSignal, QTimer
from PyQt6.QtGui import QIcon
from loguru import logger

from ..core.config.config_manager import ConfigManager
from ..core.connection.connection_manager import ClientConnectionManager
from ..core.proxy.local_proxy import LocalProxy
from common.models.tunnel import TunnelType
import asyncio


class ConnectionThread(QThread):
    """连接线程"""
    connected = pyqtSignal()
    disconnected = pyqtSignal()
    message_received = pyqtSignal(str)
    
    def __init__(self, connection_manager: ClientConnectionManager):
        super().__init__()
        self.connection_manager = connection_manager
        self.running = False
    
    def run(self):
        """运行连接"""
        self.running = True
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        
        try:
            loop.run_until_complete(self.connection_manager.start())
        except Exception as e:
            logger.error(f"连接线程出错: {e}")
        finally:
            loop.close()
    
    def stop(self):
        """停止连接"""
        self.running = False
        if self.connection_manager:
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            loop.run_until_complete(self.connection_manager.stop())
            loop.close()


class MainWindow(QMainWindow):
    """主窗口"""
    
    def __init__(self):
        super().__init__()
        self.config_manager = ConfigManager()
        self.connection_manager: ClientConnectionManager = None
        self.connection_thread: ConnectionThread = None
        self.local_proxy: LocalProxy = None
        
        self.init_ui()
        self.load_config()
    
    def init_ui(self):
        """初始化UI"""
        self.setWindowTitle("NPS 客户端")
        self.setGeometry(100, 100, 1000, 700)
        
        # 中央部件
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        
        # 主布局
        main_layout = QVBoxLayout(central_widget)
        
        # 标签页
        tabs = QTabWidget()
        tabs.addTab(self.create_connection_tab(), "连接")
        tabs.addTab(self.create_servers_tab(), "服务器")
        tabs.addTab(self.create_tunnels_tab(), "隧道")
        tabs.addTab(self.create_logs_tab(), "日志")
        
        main_layout.addWidget(tabs)
        
        # 状态栏
        self.statusBar().showMessage("未连接")
    
    def create_connection_tab(self) -> QWidget:
        """创建连接标签页"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        
        # 服务器配置组
        server_group = QGroupBox("服务器配置")
        server_layout = QVBoxLayout()
        
        server_url_layout = QVBoxLayout()
        server_url_layout.addWidget(QLabel("服务器地址（每行一个，格式：URL|名称|区域|优先级）:"))
        self.server_url_text = QTextEdit()
        self.server_url_text.setPlaceholderText("http://server1:8080|服务器1|US-West|10\nhttp://server2:8080|服务器2|US-East|5")
        self.server_url_text.setMaximumHeight(100)
        server_url_layout.addWidget(self.server_url_text)
        server_layout.addLayout(server_url_layout)
        
        server_group.setLayout(server_layout)
        layout.addWidget(server_group)
        
        # 客户端配置组
        client_group = QGroupBox("客户端配置")
        client_layout = QVBoxLayout()
        
        client_id_layout = QHBoxLayout()
        client_id_layout.addWidget(QLabel("客户端ID:"))
        self.client_id_input = QLineEdit()
        client_id_layout.addWidget(self.client_id_input)
        client_layout.addLayout(client_id_layout)
        
        client_token_layout = QHBoxLayout()
        client_token_layout.addWidget(QLabel("Token:"))
        self.client_token_input = QLineEdit()
        self.client_token_input.setEchoMode(QLineEdit.EchoMode.Password)
        client_token_layout.addWidget(self.client_token_input)
        client_layout.addLayout(client_token_layout)
        
        client_name_layout = QHBoxLayout()
        client_name_layout.addWidget(QLabel("客户端名称:"))
        self.client_name_input = QLineEdit()
        client_name_layout.addWidget(self.client_name_input)
        client_layout.addLayout(client_name_layout)
        
        client_group.setLayout(client_layout)
        layout.addWidget(client_group)
        
        # 连接按钮
        button_layout = QHBoxLayout()
        self.connect_button = QPushButton("连接")
        self.connect_button.clicked.connect(self.toggle_connection)
        self.disconnect_button = QPushButton("断开")
        self.disconnect_button.clicked.connect(self.disconnect)
        self.disconnect_button.setEnabled(False)
        button_layout.addWidget(self.connect_button)
        button_layout.addWidget(self.disconnect_button)
        layout.addLayout(button_layout)
        
        layout.addStretch()
        return widget
    
    def create_tunnels_tab(self) -> QWidget:
        """创建隧道标签页"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        
        # 隧道列表
        self.tunnels_table = QTableWidget()
        self.tunnels_table.setColumnCount(6)
        self.tunnels_table.setHorizontalHeaderLabels([
            "名称", "类型", "本地地址", "远程端口", "状态", "操作"
        ])
        layout.addWidget(self.tunnels_table)
        
        # 操作按钮
        button_layout = QHBoxLayout()
        add_button = QPushButton("添加隧道")
        add_button.clicked.connect(self.add_tunnel)
        remove_button = QPushButton("删除隧道")
        remove_button.clicked.connect(self.remove_tunnel)
        button_layout.addWidget(add_button)
        button_layout.addWidget(remove_button)
        layout.addLayout(button_layout)
        
        return widget
    
    def create_servers_tab(self) -> QWidget:
        """创建服务器状态标签页"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        
        # 服务器状态表格
        self.servers_table = QTableWidget()
        self.servers_table.setColumnCount(6)
        self.servers_table.setHorizontalHeaderLabels([
            "URL", "名称", "区域", "延迟(ms)", "状态", "错误次数"
        ])
        layout.addWidget(self.servers_table)
        
        # 刷新按钮
        refresh_button = QPushButton("刷新服务器状态")
        refresh_button.clicked.connect(self.refresh_servers_status)
        layout.addWidget(refresh_button)
        
        return widget
    
    def refresh_servers_status(self):
        """刷新服务器状态"""
        if self.connection_manager:
            statuses = self.connection_manager.get_all_servers_status()
            self.servers_table.setRowCount(len(statuses))
            
            for i, status in enumerate(statuses):
                self.servers_table.setItem(i, 0, QTableWidgetItem(status['url']))
                self.servers_table.setItem(i, 1, QTableWidgetItem(status.get('name', '')))
                self.servers_table.setItem(i, 2, QTableWidgetItem(status.get('region', '')))
                self.servers_table.setItem(i, 3, QTableWidgetItem(f"{status.get('latency', 0):.2f}"))
                self.servers_table.setItem(i, 4, QTableWidgetItem(status.get('status', 'unknown')))
                self.servers_table.setItem(i, 5, QTableWidgetItem(str(status.get('error_count', 0))))
    
    def create_logs_tab(self) -> QWidget:
        """创建日志标签页"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        
        self.logs_text = QTextEdit()
        self.logs_text.setReadOnly(True)
        layout.addWidget(self.logs_text)
        
        return widget
    
    def load_config(self):
        """加载配置"""
        server_urls = self.config_manager.get_server_urls()
        self.server_url_text.setPlainText('\n'.join(server_urls))
        self.client_id_input.setText(self.config_manager.get_client_id())
        self.client_token_input.setText(self.config_manager.get_client_token())
        self.client_name_input.setText(self.config_manager.get_client_name())
    
    def save_config(self):
        """保存配置"""
        server_text = self.server_url_text.toPlainText().strip()
        server_urls = [line.strip() for line in server_text.split('\n') if line.strip()]
        self.config_manager.set_server_urls(server_urls)
        self.config_manager.set('client.id', self.client_id_input.text())
        self.config_manager.set('client.token', self.client_token_input.text())
        self.config_manager.set('client.name', self.client_name_input.text())
    
    def toggle_connection(self):
        """切换连接状态"""
        if self.connection_thread and self.connection_thread.isRunning():
            self.disconnect()
        else:
            self.connect()
    
    def connect(self):
        """连接服务器"""
        # 保存配置
        self.save_config()
        
        server_text = self.server_url_text.toPlainText().strip()
        server_urls = [line.strip() for line in server_text.split('\n') if line.strip()]
        client_id = self.client_id_input.text()
        token = self.client_token_input.text()
        
        if not server_urls or not client_id or not token:
            QMessageBox.warning(self, "警告", "请填写完整的连接信息")
            return
        
        # 创建连接管理器（支持多服务器）
        self.connection_manager = ClientConnectionManager(
            server_urls=server_urls,
            client_id=client_id,
            token=token,
        )
        
        # 创建连接线程
        self.connection_thread = ConnectionThread(self.connection_manager)
        self.connection_thread.connected.connect(self.on_connected)
        self.connection_thread.disconnected.connect(self.on_disconnected)
        self.connection_thread.start()
        
        self.connect_button.setEnabled(False)
        self.disconnect_button.setEnabled(True)
        self.statusBar().showMessage("正在连接...")
        self.add_log("正在连接到服务器...")
    
    def disconnect(self):
        """断开连接"""
        if self.connection_thread:
            self.connection_thread.stop()
            self.connection_thread.wait()
        
        self.connect_button.setEnabled(True)
        self.disconnect_button.setEnabled(False)
        self.statusBar().showMessage("未连接")
        self.add_log("已断开连接")
    
    def on_connected(self):
        """连接成功"""
        server_info = self.connection_manager.get_current_server_info()
        if server_info:
            self.statusBar().showMessage(f"已连接到 {server_info.get('name', server_info['url'])} (延迟: {server_info.get('latency', 0):.2f}ms)")
            self.add_log(f"连接成功: {server_info['url']} (延迟: {server_info.get('latency', 0):.2f}ms)")
        else:
            self.statusBar().showMessage("已连接")
            self.add_log("连接成功")
        
        # 刷新服务器状态
        self.refresh_servers_status()
    
    def on_disconnected(self):
        """断开连接"""
        self.connect_button.setEnabled(True)
        self.disconnect_button.setEnabled(False)
        self.statusBar().showMessage("未连接")
        self.add_log("连接断开")
    
    def add_tunnel(self):
        """添加隧道"""
        # TODO: 实现添加隧道对话框
        QMessageBox.information(self, "提示", "添加隧道功能待实现")
    
    def remove_tunnel(self):
        """删除隧道"""
        # TODO: 实现删除隧道
        QMessageBox.information(self, "提示", "删除隧道功能待实现")
    
    def add_log(self, message: str):
        """添加日志"""
        self.logs_text.append(message)
    
    def closeEvent(self, event):
        """关闭事件"""
        if self.connection_thread and self.connection_thread.isRunning():
            self.disconnect()
        event.accept()


def main():
    """主函数"""
    app = QApplication(sys.argv)
    
    window = MainWindow()
    window.show()
    
    sys.exit(app.exec())


if __name__ == "__main__":
    main()

