"""
本地代理服务
"""
import asyncio
from typing import Optional, Dict
from loguru import logger

from common.models.tunnel import Tunnel, TunnelType
from ..connection.connection_manager import ClientConnectionManager


class LocalProxy:
    """本地代理"""
    
    def __init__(self, connection_manager: ClientConnectionManager):
        self.connection_manager = connection_manager
        self._proxies: Dict[str, asyncio.Task] = {}
    
    async def start_tunnel_proxy(self, tunnel: Tunnel):
        """启动隧道代理"""
        tunnel_id = tunnel.id
        
        if tunnel_id in self._proxies:
            logger.warning(f"隧道 {tunnel_id} 已启动")
            return
        
        if tunnel.tunnel_type == TunnelType.TCP:
            task = asyncio.create_task(self._tcp_proxy(tunnel))
        elif tunnel.tunnel_type == TunnelType.HTTP:
            task = asyncio.create_task(self._http_proxy(tunnel))
        elif tunnel.tunnel_type == TunnelType.UDP:
            task = asyncio.create_task(self._udp_proxy(tunnel))
        else:
            logger.error(f"不支持的隧道类型: {tunnel.tunnel_type}")
            return
        
        self._proxies[tunnel_id] = task
        logger.info(f"启动隧道代理: {tunnel_id} ({tunnel.tunnel_type.value})")
    
    async def stop_tunnel_proxy(self, tunnel_id: str):
        """停止隧道代理"""
        if tunnel_id in self._proxies:
            task = self._proxies[tunnel_id]
            task.cancel()
            try:
                await task
            except asyncio.CancelledError:
                pass
            del self._proxies[tunnel_id]
            logger.info(f"停止隧道代理: {tunnel_id}")
    
    async def _tcp_proxy(self, tunnel: Tunnel):
        """TCP代理"""
        try:
            while True:
                try:
                    # 监听本地端口
                    server = await asyncio.start_server(
                        lambda r, w: self._handle_tcp_connection(tunnel.id, r, w),
                        tunnel.local_host,
                        tunnel.local_port
                    )
                    logger.info(f"TCP代理监听: {tunnel.local_host}:{tunnel.local_port}")
                    
                    async with server:
                        await server.serve_forever()
                        
                except asyncio.CancelledError:
                    raise
                except Exception as e:
                    logger.error(f"TCP代理出错: {e}")
                    await asyncio.sleep(5)
                    
        except asyncio.CancelledError:
            logger.info(f"TCP代理已取消: {tunnel.id}")
    
    async def _handle_tcp_connection(self, tunnel_id: str, reader: asyncio.StreamReader, writer: asyncio.StreamWriter):
        """处理TCP连接"""
        try:
            # 读取本地数据并转发到服务器
            while True:
                data = await reader.read(4096)
                if not data:
                    break
                
                await self.connection_manager.send_tunnel_data(tunnel_id, data)
                
        except Exception as e:
            logger.error(f"处理TCP连接失败: {e}")
        finally:
            writer.close()
            await writer.wait_closed()
    
    async def _http_proxy(self, tunnel: Tunnel):
        """HTTP代理"""
        try:
            while True:
                try:
                    # 监听本地端口
                    server = await asyncio.start_server(
                        lambda r, w: self._handle_http_connection(tunnel.id, r, w),
                        tunnel.local_host,
                        tunnel.local_port
                    )
                    logger.info(f"HTTP代理监听: {tunnel.local_host}:{tunnel.local_port}")
                    
                    async with server:
                        await server.serve_forever()
                        
                except asyncio.CancelledError:
                    raise
                except Exception as e:
                    logger.error(f"HTTP代理出错: {e}")
                    await asyncio.sleep(5)
                    
        except asyncio.CancelledError:
            logger.info(f"HTTP代理已取消: {tunnel.id}")
    
    async def _handle_http_connection(self, tunnel_id: str, reader: asyncio.StreamReader, writer: asyncio.StreamWriter):
        """处理HTTP连接"""
        try:
            # 读取HTTP请求
            request_data = b""
            while True:
                chunk = await reader.read(4096)
                if not chunk:
                    break
                request_data += chunk
                # 检查是否收到完整的HTTP请求头
                if b"\r\n\r\n" in request_data:
                    break
            
            if request_data:
                # 转发到服务器
                await self.connection_manager.send_tunnel_data(tunnel_id, request_data)
                
        except Exception as e:
            logger.error(f"处理HTTP连接失败: {e}")
        finally:
            writer.close()
            await writer.wait_closed()
    
    async def _udp_proxy(self, tunnel: Tunnel):
        """UDP代理"""
        import socket
        
        try:
            # 创建UDP socket
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            sock.bind((tunnel.local_host, tunnel.local_port))
            sock.setblocking(False)
            
            logger.info(f"UDP代理监听: {tunnel.local_host}:{tunnel.local_port}")
            
            loop = asyncio.get_event_loop()
            
            while True:
                try:
                    # 接收UDP数据
                    data, addr = await loop.sock_recvfrom(sock, 4096)
                    if data:
                        # 转发到服务器（简化实现，实际需要处理UDP的特殊性）
                        await self.connection_manager.send_tunnel_data(tunnel.id, data)
                        
                except asyncio.CancelledError:
                    raise
                except Exception as e:
                    logger.error(f"UDP代理出错: {e}")
                    await asyncio.sleep(0.1)
                    
        except asyncio.CancelledError:
            logger.info(f"UDP代理已取消: {tunnel.id}")
        finally:
            try:
                sock.close()
            except:
                pass

